<?php

namespace wbb\page;

use wbb\data\board\Board;
use wbb\data\thread\ViewableThreadList;
use wbb\system\WBBCore;
use wcf\data\user\User;
use wcf\page\MultipleLinkPage;
use wcf\system\cache\runtime\UserRuntimeCache;
use wcf\system\clipboard\ClipboardHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\language\LanguageFactory;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;

/**
 * Shows  a list of posts by a certain user.
 *
 * @author      Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license     WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package     WoltLabSuite\Forum\Page
 * @since       5.2
 *
 * @property    ViewableThreadList      $objectList
 */
class UserThreadListPage extends MultipleLinkPage
{
    /**
     * @inheritDoc
     */
    public $itemsPerPage = WBB_BOARD_THREADS_PER_PAGE;

    /**
     * @inheritDoc
     */
    public $sqlOrderBy = 'thread.time DESC, thread.threadID DESC';

    /**
     * @inheritDoc
     */
    public $objectListClassName = ViewableThreadList::class;

    /**
     * user id
     * @var int
     */
    public $userID = 0;

    /**
     * user object
     * @var User
     */
    public $user;

    /**
     * @inheritDoc
     */
    public function readParameters()
    {
        parent::readParameters();

        if (isset($_REQUEST['id'])) {
            $this->userID = \intval($_REQUEST['id']);
        }
        $this->user = UserRuntimeCache::getInstance()->getObject($this->userID);
        if ($this->user === null) {
            throw new IllegalLinkException();
        }

        $this->canonicalURL = LinkHandler::getInstance()->getLink('UserThreadList', [
            'application' => 'wbb',
            'object' => $this->user,
        ], ($this->pageNo ? 'pageNo=' . $this->pageNo : ''));

        /** @noinspection PhpUndefinedFieldInspection */
        if (WCF::getUser()->threadsPerPage) {
            /** @noinspection PhpUndefinedFieldInspection */
            $this->itemsPerPage = WCF::getUser()->threadsPerPage;
        }
    }

    /**
     * @inheritDoc
     */
    protected function initObjectList()
    {
        parent::initObjectList();

        $this->objectList->getConditionBuilder()->add('thread.userID = ?', [$this->user->userID]);
        if (!WCF::getSession()->getPermission('mod.board.canEnablePost')) {
            if (WCF::getUser()->userID) {
                $this->objectList->getConditionBuilder()->add(
                    '(thread.isDisabled = 0 OR thread.userID = ?)',
                    [WCF::getUser()->userID]
                );
            } else {
                $this->objectList->getConditionBuilder()->add('thread.isDisabled = 0');
            }
        }
        if (!WCF::getSession()->getPermission('mod.board.canReadDeletedPost')) {
            $this->objectList->getConditionBuilder()->add('thread.isDeleted = 0');
        }

        $boardIDs = Board::getAccessibleBoardIDs(['canViewBoard', 'canEnterBoard', 'canReadThread']);
        $privateBoardIDs = Board::getPrivateBoardIDs(['canViewBoard', 'canEnterBoard', 'canReadThread']);
        if (empty($boardIDs) && empty($privateBoardIDs)) {
            $this->objectList->getConditionBuilder()->add("1=0");
        } else {
            if (empty($privateBoardIDs)) {
                $this->objectList->getConditionBuilder()->add("thread.boardID IN (?)", [$boardIDs]);
            } elseif (empty($boardIDs)) {
                $this->objectList->getConditionBuilder()->add(
                    "(thread.boardID IN (?) AND thread.userID = ?)",
                    [$privateBoardIDs, WCF::getUser()->userID]
                );
            } else {
                $this->objectList->getConditionBuilder()->add(
                    "(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))",
                    [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]
                );
            }
        }

        // apply language filter
        if (LanguageFactory::getInstance()->multilingualismEnabled() && \count(WCF::getUser()->getLanguageIDs())) {
            $this->objectList->getConditionBuilder()->add(
                '(thread.languageID IN (?) OR thread.languageID IS NULL)',
                [WCF::getUser()->getLanguageIDs()]
            );
        }
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        parent::readData();

        // add breadcrumbs
        WBBCore::getInstance()->setLocation();
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        WCF::getTPL()->assign([
            'hasMarkedItems' => ClipboardHandler::getInstance()->hasMarkedItems(
                ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.thread')
            ),
            'userID' => $this->userID,
            'user' => $this->user,
        ]);
    }
}
