<?php

namespace wbb\system\form\builder\field\board;

use wbb\data\board\BoardNode;
use wbb\data\board\BoardNodeList;
use wcf\system\form\builder\data\processor\CustomFormDataProcessor;
use wcf\system\form\builder\exception\InvalidFormFieldValue;
use wcf\system\form\builder\field\AbstractFormField;
use wcf\system\form\builder\field\IAttributeFormField;
use wcf\system\form\builder\field\ICssClassFormField;
use wcf\system\form\builder\field\IImmutableFormField;
use wcf\system\form\builder\field\TCssClassFormField;
use wcf\system\form\builder\field\TImmutableFormField;
use wcf\system\form\builder\field\TInputAttributeFormField;
use wcf\system\form\builder\field\validation\FormFieldValidationError;
use wcf\system\form\builder\IFormDocument;

/**
 * Implementation of a form field for selecting multiple forums.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2021 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Form\Builder\Field\Board
 * @since   5.5
 */
final class MultipleBoardSelectionFormField extends AbstractFormField implements
    IAttributeFormField,
    ICssClassFormField,
    IImmutableFormField
{
    use TInputAttributeFormField;
    use TCssClassFormField;
    use TImmutableFormField;

    /**
     * board node list whose boards will be shown in the selection list
     * @var BoardNodeList|null
     */
    protected $boardNodeList;

    /**
     * is `true` if categories can be selected and `false` otherwise
     */
    protected $categoriesSelectable = true;

    /**
     * @inheritDoc
     */
    protected $javaScriptDataHandlerModule = 'WoltLabSuite/Core/Form/Builder/Field/Checkboxes';

    /**
     * is `true` if externals links are shown in the board list and `false` otherwise
     */
    protected $supportExternalLinks = true;

    /**
     * @inheritDoc
     */
    protected $templateName = '__multipleBoardSelectionFormField';

    /**
     * @inheritDoc
     */
    protected $templateApplication = 'wbb';

    /**
     * @inheritDoc
     */
    protected $value = [];

    /**
     * Returns `true` if categories can be selected and `false` otherwise.
     */
    public function areCategoriesSelectable(): bool
    {
        return $this->categoriesSelectable;
    }

    /**
     * Sets the board node list whose boards will be shown in the selection list and returns this field.
     *
     * This method itself calls `BoardNodeList::readNodeTree()` so that in most cases,
     * `new BoardNodeList()` can directly be passed to this method without having to call
     * `BoardNodeList::readNodeTree()` manually.
     */
    public function boardNodeList(BoardNodeList $boardNodeList): self
    {
        $this->boardNodeList = $boardNodeList;
        $this->boardNodeList->readNodeTree();

        return $this;
    }

    /**
     * Sets if categories can be selected and returns this field.
     */
    public function categoriesSelectable(bool $categoriesSelectable = true): self
    {
        $this->categoriesSelectable = $categoriesSelectable;

        return $this;
    }

    /**
     * Returns the board node list whose boards will be shown in the selection list.
     */
    public function getBoardNodeList(): BoardNodeList
    {
        if ($this->boardNodeList === null) {
            throw new \UnexpectedValueException(
                "Board node list has not been set for form field '{$this->getPrefixedId()}'."
            );
        }

        return $this->boardNodeList;
    }

    /**
     * @inheritDoc
     */
    public function getHtml()
    {
        if ($this->boardNodeList === null) {
            throw new \UnexpectedValueException(
                "Board node list has not been set for form field '{$this->getPrefixedId()}'."
            );
        }

        return parent::getHtml();
    }

    /**
     * Returns the ids of the boards from the set board node list that are selectable.
     */
    protected function getSelectableBoardIds(): array
    {
        $boardIds = [];
        /** @var BoardNode $boardNode */
        foreach ($this->getBoardNodeList()->getNodeList() as $boardNode) {
            if (!$this->areCategoriesSelectable() && $boardNode->getBoard()->isCategory()) {
                continue;
            } elseif (!$this->supportsExternalLinks() && $boardNode->getBoard()->isExternalLink()) {
                continue;
            }

            $boardIds[] = $boardNode->getObjectID();
        }

        return $boardIds;
    }

    /**
     * @inheritDoc
     */
    public function hasSaveValue()
    {
        return false;
    }

    /**
     * @inheritDoc
     */
    public function populate()
    {
        parent::populate();

        $this->getDocument()->getDataHandler()->addProcessor(
            new CustomFormDataProcessor(
                'multiple',
                function (IFormDocument $document, array $parameters) {
                    if ($this->checkDependencies() && !empty($this->getValue())) {
                        $parameters[$this->getObjectProperty()] = $this->getValue();
                    }

                    return $parameters;
                }
            )
        );

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function readValue()
    {
        if ($this->getDocument()->hasRequestData($this->getPrefixedId())) {
            $value = $this->getDocument()->getRequestData($this->getPrefixedId());

            if (\is_array($value)) {
                $this->value = $value;
            }
        }

        return $this;
    }

    /**
     * Sets if externals links are shown in the board list and returns this field.
     */
    public function supportExternalLinks(bool $supportExternalLinks): self
    {
        $this->supportExternalLinks = $supportExternalLinks;

        return $this;
    }

    /**
     * Returns `true` if externals links are shown in the board list and `false` otherwise.
     */
    public function supportsExternalLinks(): bool
    {
        return $this->supportExternalLinks;
    }

    /**
     * @inheritDoc
     */
    public function validate()
    {
        $value = $this->getValue();

        if (empty($value) && $this->isRequired()) {
            $this->addValidationError(new FormFieldValidationError('empty'));
        } elseif ($value !== null && !empty(\array_diff($this->getValue(), $this->getSelectableBoardIds()))) {
            $this->addValidationError(
                new FormFieldValidationError(
                    'invalidValue',
                    'wcf.global.form.error.noValidSelection'
                )
            );
        }

        parent::validate();
    }

    /**
     * @inheritDoc
     */
    public function value($value)
    {
        // Ignore `null` as value which can be passed either for nullable fields or as value if no
        // options are available
        if ($value === null) {
            return $this;
        }

        if (!\is_array($value)) {
            throw new InvalidFormFieldValue($this, 'array', \gettype($value));
        }

        $unknownValues = \array_diff($value, $this->getSelectableBoardIds());
        if (!empty($unknownValues)) {
            throw new \InvalidArgumentException(
                "Unknown values '" . \implode("', '", $unknownValues) . "' for field '{$this->getId()}'."
            );
        }

        return parent::value($value);
    }
}
