<?php

namespace wbb\data\modification\log;

use wbb\data\post\PostList;
use wbb\data\thread\Thread;
use wbb\system\log\modification\PostModificationLogHandler;
use wbb\system\log\modification\ThreadModificationLogHandler;
use wcf\data\label\Label;
use wcf\data\modification\log\ModificationLogList;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\label\LabelHandler;
use wcf\system\WCF;

/**
 * Represents a list of modification logs for thread log page.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Modification\Log
 *
 * @method  ViewablePostModificationLog|ViewableThreadModificationLog       current()
 * @method  ViewablePostModificationLog[]|ViewableThreadModificationLog[]       getObjects()
 * @method  ViewablePostModificationLog|ViewableThreadModificationLog|null      getSingleObject()
 * @method  ViewablePostModificationLog|ViewableThreadModificationLog|null      search($objectID)
 * @property    ViewablePostModificationLog[]|ViewableThreadModificationLog[]       $objects
 */
class ThreadLogModificationLogList extends ModificationLogList
{
    /**
     * post object type id
     * @var int
     */
    public $postObjectTypeID = 0;

    /**
     * thread object type id
     * @var int
     */
    public $threadObjectTypeID = 0;

    /**
     * thread object
     * @var Thread
     */
    public $thread;

    /**
     * @inheritDoc
     */
    public function __construct()
    {
        parent::__construct();

        // get object types
        $this->postObjectTypeID = PostModificationLogHandler::getInstance()->getObjectType()->objectTypeID;
        $this->threadObjectTypeID = ThreadModificationLogHandler::getInstance()->getObjectType()->objectTypeID;
    }

    /**
     * Initializes the thread log modification log list.
     *
     * @param   Thread      $thread
     */
    public function setThread(Thread $thread)
    {
        $this->thread = $thread;
    }

    /**
     * @inheritDoc
     */
    public function countObjects()
    {
        $sql = "SELECT  COUNT(modification_log.logID) AS count
                FROM    wcf" . WCF_N . "_modification_log modification_log
                WHERE   modification_log.objectTypeID = ?
                    AND modification_log.objectID = ?
                UNION
                SELECT  COUNT(modification_log.logID) AS count
                FROM    wcf" . WCF_N . "_modification_log modification_log
                WHERE   modification_log.objectTypeID = ?
                    AND modification_log.parentObjectID = ?";
        $statement = WCF::getDB()->prepareStatement($sql);
        $statement->execute([
            $this->threadObjectTypeID,
            $this->thread->threadID,
            $this->postObjectTypeID,
            $this->thread->threadID,
        ]);
        $count = 0;
        while ($row = $statement->fetchArray()) {
            $count += $row['count'];
        }

        return $count;
    }

    /**
     * @inheritDoc
     */
    public function readObjects()
    {
        $sql = "SELECT  modification_log.*
                FROM    wcf" . WCF_N . "_modification_log modification_log
                WHERE   modification_log.objectTypeID = ?
                    AND modification_log.objectID = ?
                UNION
                SELECT  modification_log.*
                FROM    wcf" . WCF_N . "_modification_log modification_log
                WHERE   modification_log.objectTypeID = ?
                    AND modification_log.parentObjectID = ?
                " . (!empty($this->sqlOrderBy) ? "ORDER BY " . $this->sqlOrderBy : '');
        $statement = WCF::getDB()->prepareStatement($sql, $this->sqlLimit, $this->sqlOffset);
        $statement->execute([
            $this->threadObjectTypeID,
            $this->thread->threadID,
            $this->postObjectTypeID,
            $this->thread->threadID,
        ]);
        $this->objects = $statement->fetchObjects($this->objectClassName ?: $this->className);

        $postIDs = $userIDs = [];
        foreach ($this->objects as $key => &$object) {
            if ($object->userID) {
                $userIDs[] = $object->userID;
            }

            if ($object->objectTypeID == $this->postObjectTypeID) {
                $object = new ViewablePostModificationLog($object);
                $postIDs[] = $object->objectID;
            } else {
                if ($object->action == 'setLabel') {
                    // check permissions
                    if ($object->label instanceof Label) {
                        $labelID = $object->label->labelID;
                    } elseif ($object->oldLabel instanceof Label) {
                        $labelID = $object->oldLabel->labelID;
                    } else {
                        $labelID = false;
                    }

                    if (!$labelID || !LabelHandler::getInstance()->validateCanView([$labelID])[$labelID]) {
                        unset($this->objects[$key]);
                        continue;
                    }
                }

                $object = new ViewableThreadModificationLog($object);
            }
        }
        unset($object);

        // use table index as array index
        $objects = [];
        foreach ($this->objects as $object) {
            $objectID = $object->{$this->getDatabaseTableIndexName()};
            $objects[$objectID] = $object;

            $this->indexToObject[] = $objectID;
        }
        $this->objectIDs = $this->indexToObject;
        $this->objects = $objects;

        if (!empty($userIDs)) {
            UserProfileRuntimeCache::getInstance()->cacheObjectIDs($userIDs);
        }

        if (!empty($postIDs)) {
            $postList = new PostList();
            $postList->setObjectIDs($postIDs);
            $postList->readObjects();

            foreach ($this->objects as $object) {
                if ($object instanceof ViewablePostModificationLog) {
                    foreach ($postList as $post) {
                        if ($object->objectID == $post->postID) {
                            $object->setPost($post);
                            break;
                        }
                    }
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function getDatabaseTableAlias()
    {
        return '';
    }
}
