<?php

namespace wbb\system\page\handler;

use wbb\data\board\Board;
use wbb\data\thread\ViewableThread;
use wbb\system\cache\runtime\ThreadRuntimeCache;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\language\LanguageFactory;
use wcf\system\page\handler\AbstractLookupPageHandler;
use wcf\system\page\handler\IOnlineLocationPageHandler;
use wcf\system\search\SearchEngine;
use wcf\system\WCF;

/**
 * Page handler for threads.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Page\Handler
 * @since   5.0
 */
class ThreadPageHandler extends AbstractLookupPageHandler implements IOnlineLocationPageHandler
{
    use TThreadOnlineLocationPageHandler;

    /**
     * @inheritDoc
     */
    public function getLink($objectID)
    {
        return ThreadRuntimeCache::getInstance()->getObject($objectID)->getLink();
    }

    /**
     * @inheritDoc
     */
    public function getOutstandingItemCount($objectID = null)
    {
        // this is not supported for threads due to potential performance bottlenecks
        // when trying to return the actual number of new replies
        return 0;
    }

    /**
     * @inheritDoc
     */
    public function isValid($objectID)
    {
        return ThreadRuntimeCache::getInstance()->getObject($objectID) !== null;
    }

    /**
     * @inheritDoc
     */
    public function isVisible($objectID = null)
    {
        return ThreadRuntimeCache::getInstance()->getObject($objectID)->canRead();
    }

    /**
     * @inheritDoc
     */
    public function lookup($searchString)
    {
        $searchIndexCondition = null;

        // filter by user's content languages
        if (WCF::getUser()->userID && !empty(LanguageFactory::getInstance()->getContentLanguages())) {
            $className = SearchEngine::getInstance()->getConditionBuilderClassName();

            /** @var PreparedStatementConditionBuilder $searchIndexCondition */
            $searchIndexCondition = new $className(false);
            $searchIndexCondition->add("languageID IN (?)", WCF::getUser()->getLanguageIDs());
        }

        $innerJoin = SearchEngine::getInstance()->getInnerJoin(
            'com.woltlab.wbb.post',
            $searchString,
            true,
            $searchIndexCondition,
            'relevance DESC',
            1000
        );

        // get thread ids
        $conditionBuilder = new PreparedStatementConditionBuilder();
        $conditionBuilder->add('post.isDisabled = 0 AND post.isDeleted = 0');
        // private boards are excluded
        $conditionBuilder->add('thread.boardID IN (?)', [Board::getAccessibleBoardIDs()]);

        $sql = "SELECT      thread.threadID, search_index.relevance AS relevance
                FROM        wbb" . WCF_N . "_post post
                INNER JOIN  (" . $innerJoin['sql'] . ") search_index
                ON          post.postID = search_index.objectID
                LEFT JOIN   wbb" . WCF_N . "_thread thread
                ON          thread.threadID = post.threadID
                " . $conditionBuilder . "
                ORDER BY    relevance DESC";
        $statement = WCF::getDB()->prepareStatement($sql, 10);

        $parameters = [];
        if ($innerJoin['fulltextCondition'] !== null) {
            /** @noinspection PhpUndefinedMethodInspection */
            $parameters = $innerJoin['fulltextCondition']->getParameters();
        }
        if ($innerJoin['searchIndexCondition'] !== null) {
            /** @noinspection PhpUndefinedMethodInspection */
            $parameters = \array_merge($parameters, $innerJoin['searchIndexCondition']->getParameters());
        }
        $parameters = \array_merge($parameters, $conditionBuilder->getParameters());

        $statement->execute($parameters);
        $threadIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);

        if (empty($threadIDs)) {
            return [];
        }

        $threads = ThreadRuntimeCache::getInstance()->getObjects($threadIDs);
        $results = [];
        foreach ($threads as $thread) {
            $thread = new ViewableThread($thread);

            $description = '';
            foreach ($thread->getBoard()->getParentBoards() as $parentBoard) {
                $description .= $parentBoard->getTitle() . ' &raquo; ';
            }
            $description .= $thread->getBoard()->getTitle() . ' &raquo; ';

            $results[] = [
                'description' => $description,
                'image' => $thread->getUserProfile()->getAvatar()->getImageTag(48),
                'link' => $thread->getLink(),
                'objectID' => $thread->threadID,
                'title' => $thread->topic,
            ];
        }

        return $results;
    }
}
