<?php

namespace wbb\data\board;

/**
 * Represents an uncached board list.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Board
 */
class RealtimeBoardNodeList extends BoardNodeList
{
    /**
     * list of boards
     * @var Board[]
     */
    public $boards = [];

    /**
     * board structure
     * @var int[][]
     */
    public $boardStructure = [];

    /**
     * @inheritDoc
     */
    public function __construct($parentID = null, $startDepth = 0, array $filter = [])
    {
        parent::__construct($parentID, $startDepth, $filter);

        $boardList = new BoardList();
        $boardList->sqlOrderBy = "board.parentID ASC, board.position ASC";
        $boardList->readObjects();

        foreach ($boardList as $board) {
            $this->boards[$board->boardID] = $board;

            if (!isset($this->boardStructure[$board->parentID])) {
                $this->boardStructure[$board->parentID] = [];
            }
            $this->boardStructure[$board->parentID][] = $board->boardID;
        }
    }

    /**
     * @inheritDoc
     */
    protected function generateNodeTree($parentID, ?BoardNode $parentNode = null, array $filter = [])
    {
        $nodes = [];

        // apply filter
        $boardIDs = $this->getChildIDs($parentID);
        if (\count($filter)) {
            $boardIDs = \array_intersect($boardIDs, $filter);
        }

        foreach ($boardIDs as $boardID) {
            $board = $this->getBoard($boardID);
            if ($this->isVisible($board)) {
                /** @var BoardNode $node */
                $node = new $this->nodeClassname(
                    $parentNode,
                    $board,
                    $parentNode !== null ? $parentNode->getDepth() + 1 : 0
                );
                $nodes[] = $node;

                // get children
                $node->setChildren($this->generateNodeTree($boardID, $node));
            }
        }

        return $nodes;
    }

    /**
     * Returns a list of child board ids.
     *
     * @param   int     $parentID
     * @return  int[]
     */
    protected function getChildIDs($parentID)
    {
        return $this->boardStructure[$parentID] ?? [];
    }

    /**
     * Returns a board by id.
     *
     * @param   int         $boardID
     * @return  Board
     */
    protected function getBoard($boardID)
    {
        return $this->boards[$boardID];
    }
}
