<?php

namespace wbb\system\page\handler;

use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wcf\system\page\handler\AbstractLookupPageHandler;
use wcf\system\page\handler\IOnlineLocationPageHandler;
use wcf\system\WCF;

/**
 * Page handler for boards.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Page\Handler
 * @since   5.0
 */
class BoardPageHandler extends AbstractLookupPageHandler implements IOnlineLocationPageHandler
{
    use TBoardOnlineLocationPageHandler;

    /**
     * @inheritDoc
     */
    public function getLink($objectID)
    {
        return BoardCache::getInstance()->getBoard($objectID)->getLink();
    }

    /**
     * @inheritDoc
     */
    public function getOutstandingItemCount($objectID = null)
    {
        $boardIDs = BoardCache::getInstance()->getAllChildIDs($objectID);
        $boardIDs[] = $objectID;
        $boardIDs = \array_intersect(
            $boardIDs,
            Board::filterBoardIDs(
                \array_merge(
                    Board::getAccessibleBoardIDs(),
                    Board::getPrivateBoardIDs()
                )
            )
        );

        $count = 0;
        foreach ($boardIDs as $boardID) {
            $count += BoardCache::getInstance()->getUnreadThreads($boardID);
        }

        return $count;
    }

    /**
     * @inheritDoc
     */
    public function isValid($objectID)
    {
        return BoardCache::getInstance()->getBoard($objectID) !== null;
    }

    /**
     * @inheritDoc
     */
    public function isVisible($objectID = null)
    {
        $board = BoardCache::getInstance()->getBoard($objectID);

        return $board !== null && $board->canEnter();
    }

    /**
     * @inheritDoc
     */
    public function lookup($searchString)
    {
        $sql = "(
                    SELECT  boardID, title, title AS name
                    FROM    wbb" . WCF_N . "_board
                    WHERE   title LIKE ?
                ) UNION (
                    SELECT  0 AS boardID, languageItemValue AS title, languageItem AS name
                    FROM    wcf" . WCF_N . "_language_item
                    WHERE   languageID = ?
                        AND languageItem LIKE ?
                        AND languageItemValue LIKE ?
                )";
        $statement = WCF::getDB()->prepareStatement($sql, 10);
        $statement->execute([
            '%' . $searchString . '%',
            WCF::getLanguage()->languageID,
            'wbb.board.board%',
            '%' . $searchString . '%',
        ]);

        $results = [];
        while ($row = $statement->fetchArray()) {
            $boardID = $row['boardID'];
            if ($boardID) {
                if ($row['title'] === 'wbb.board.board' . $boardID) {
                    continue;
                }
            } else {
                if (\preg_match('~^wbb\.board\.board(\d+)$~', $row['name'], $matches)) {
                    $boardID = $matches[1];
                } else {
                    continue;
                }
            }

            $board = BoardCache::getInstance()->getBoard($boardID);

            // build hierarchy
            $description = '';
            /** @var \wbb\data\board\Board $parentBoard */
            foreach ($board->getParentBoards() as $parentBoard) {
                $description .= $parentBoard->getTitle() . ' &raquo; ';
            }

            $results[] = [
                'description' => $description,
                'image' => 'fa-folder-open-o',
                'link' => $board->getLink(),
                'objectID' => $boardID,
                'title' => $row['title'],
            ];
        }

        return $results;
    }
}
