<?php

namespace wbb\data\board;

use wbb\data\thread\ViewableThread;
use wbb\data\thread\ViewableThreadList;
use wbb\system\board\status\BoardStatusHandler;
use wbb\system\cache\builder\BoardCacheBuilder;
use wbb\system\cache\builder\BoardDataCacheBuilder;
use wbb\system\thread\status\ThreadStatusHandler;
use wcf\data\page\PageCache;
use wcf\data\user\ignore\UserIgnore;
use wcf\data\user\online\UserOnline;
use wcf\data\user\online\UsersOnlineList;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\language\LanguageFactory;
use wcf\system\SingletonFactory;
use wcf\system\user\collapsible\content\UserCollapsibleContentHandler;
use wcf\system\user\UserProfileHandler;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\WCF;

/**
 * Manages the board cache.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Board
 */
class BoardCache extends SingletonFactory
{
    /**
     * cached board structure
     * @var array
     */
    protected $cachedBoardStructure = [];

    /**
     * cached boards
     * @var Board[]
     */
    protected $cachedBoards = [];

    /**
     * cached label groups
     * @var int[][]
     */
    protected $cachedLabelGroups = [];

    /**
     * list of cache moderator user ids per board id
     * @var mixed[][]
     */
    protected $cachedModerators = [];

    /**
     * cached board counter
     * @var array
     */
    protected $counts = [];

    /**
     * cached last posts (thread ids)
     * @var int[]
     */
    protected $lastPostThreadIDs = [];

    /**
     * cached last posts
     * @var ViewableThread[]
     */
    protected $lastPosts;

    /**
     * list of closed boards
     * @var int[]
     */
    protected $closedBoardIDs;

    /**
     * list of ignored boards
     * @var int[]
     */
    protected $ignoredBoardIDs;

    /**
     * number of unread threads
     * @var int[]
     */
    protected $unreadThreads;

    /**
     * users online
     * @var UserOnline[][]
     */
    protected $usersOnline;

    /**
     * @inheritDoc
     */
    protected function init()
    {
        // get board cache
        $this->cachedBoardStructure = BoardCacheBuilder::getInstance()->getData([], 'boardStructure');
        $this->cachedBoards = BoardCacheBuilder::getInstance()->getData([], 'boards');
        $this->cachedLabelGroups = BoardCacheBuilder::getInstance()->getData([], 'labelGroups');
        $this->cachedModerators = BoardCacheBuilder::getInstance()->getData([], 'moderators');

        // get board data cache
        $this->counts = BoardDataCacheBuilder::getInstance()->getData([], 'counts');
        $this->lastPostThreadIDs = BoardDataCacheBuilder::getInstance()->getData([], 'lastPostThreadIDs');
    }

    /**
     * Calculates the number of unread threads.
     */
    protected function initUnreadThreads()
    {
        $this->unreadThreads = [];
        if (WCF::getUser()->userID) {
            $conditionBuilder = new PreparedStatementConditionBuilder();
            $conditionBuilder->add(
                'thread.lastPostTime > ?',
                [VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread')]
            );
            $conditionBuilder->add(
                'thread.isDeleted = 0 AND thread.isDisabled = 0 AND thread.movedThreadID IS NULL'
            );
            $conditionBuilder->add(
                '(thread.lastPostTime > tracked_thread_visit.visitTime OR tracked_thread_visit.visitTime IS NULL)'
            );
            $conditionBuilder->add(
                '(thread.lastPostTime > tracked_board_visit.visitTime OR tracked_board_visit.visitTime IS NULL)'
            );

            // apply language filter
            if (LanguageFactory::getInstance()->multilingualismEnabled() && \count(WCF::getUser()->getLanguageIDs())) {
                $conditionBuilder->add(
                    '(thread.languageID IN (?) OR thread.languageID IS NULL)',
                    [WCF::getUser()->getLanguageIDs()]
                );
            }

            // get board ids
            $boardIDs = Board::filterBoardIDs(Board::getAccessibleBoardIDs());
            $privateBoardIDs = Board::filterBoardIDs(Board::getPrivateBoardIDs());

            if (empty($boardIDs) && empty($privateBoardIDs)) {
                return;
            } else {
                if (empty($privateBoardIDs)) {
                    $conditionBuilder->add("thread.boardID IN (?)", [$boardIDs]);
                } elseif (empty($boardIDs)) {
                    $conditionBuilder->add(
                        "(thread.boardID IN (?) AND thread.userID = ?)",
                        [$privateBoardIDs, WCF::getUser()->userID]
                    );
                } else {
                    $conditionBuilder->add(
                        "(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))",
                        [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]
                    );
                }
            }

            $ignoredUsers = UserProfileHandler::getInstance()->getIgnoredUsers(UserIgnore::TYPE_HIDE_MESSAGES);
            if (!empty($ignoredUsers)) {
                $conditionBuilder->add(
                    "(thread.userID IS NULL OR thread.userID NOT IN (?))",
                    [$ignoredUsers]
                );
            }

            // Exclude ignored threads.
            $conditionBuilder->add(
                "(thread_user_status.status <> ? OR thread_user_status.status IS NULL)",
                [ThreadStatusHandler::SUBSCRIPTION_MODE_IGNORING]
            );

            $sql = "SELECT      COUNT(*) AS count, thread.boardID
                    FROM        wbb" . WCF_N . "_thread thread
                    LEFT JOIN   wcf" . WCF_N . "_tracked_visit tracked_thread_visit
                    ON          tracked_thread_visit.objectTypeID = ?
                            AND tracked_thread_visit.objectID = thread.threadID
                            AND tracked_thread_visit.userID = ?
                    LEFT JOIN   wcf" . WCF_N . "_tracked_visit tracked_board_visit
                    ON          tracked_board_visit.objectTypeID = ?
                            AND tracked_board_visit.objectID = thread.boardID
                            AND tracked_board_visit.userID = ?
                    LEFT JOIN   wbb" . WCF_N . "_thread_user_status thread_user_status
                    ON          thread_user_status.userID = ?
                            AND thread_user_status.threadID = thread.threadID
                    " . $conditionBuilder . "
                    GROUP BY    thread.boardID";
            $statement = WCF::getDB()->prepareStatement($sql);
            $statement->execute(\array_merge(
                [
                    VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.thread'),
                    WCF::getUser()->userID,
                    VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.board'),
                    WCF::getUser()->userID,
                    WCF::getUser()->userID,
                ],
                $conditionBuilder->getParameters()
            ));
            $this->unreadThreads = $statement->fetchMap('boardID', 'count');
        }
    }

    /**
     * Reads the users online.
     */
    protected function initUsersOnline()
    {
        $this->usersOnline = [];

        $usersOnlineList = new UsersOnlineList();
        $usersOnlineList->getConditionBuilder()->add(
            '(session.pageID = ? OR session.parentPageID = ?)',
            [PageCache::getInstance()->getPageByIdentifier('com.woltlab.wbb.Board')->pageID, PageCache::getInstance()->getPageByIdentifier('com.woltlab.wbb.Board')->pageID]
        );
        $usersOnlineList->getConditionBuilder()->add('session.userID IS NOT NULL');
        $usersOnlineList->readObjects();

        foreach ($usersOnlineList as $user) {
            $boardID = ($user->pageID == PageCache::getInstance()->getPageByIdentifier('com.woltlab.wbb.Board')->pageID ? $user->pageObjectID : $user->parentPageObjectID);
            if (!isset($this->usersOnline[$boardID])) {
                $this->usersOnline[$boardID] = [];
            }

            $this->usersOnline[$boardID][] = $user;
        }
    }

    /**
     * Returns the board with the given board id from cache or `null` if no such board exists.
     *
     * @param   int     $boardID
     * @return  Board|null
     */
    public function getBoard($boardID)
    {
        return $this->cachedBoards[$boardID] ?? null;
    }

    /**
     * Returns the direct children of a board.
     *
     * @param   int     $parentID
     * @return  int[]
     */
    public function getChildIDs($parentID = null)
    {
        if ($parentID === null) {
            $parentID = '';
        }

        if (!isset($this->cachedBoardStructure[$parentID])) {
            return [];
        }

        return $this->cachedBoardStructure[$parentID];
    }

    /**
     * Returns all children of a board (recursively).
     *
     * @param   int     $parentID
     * @return  int[]
     */
    public function getAllChildIDs($parentID = null)
    {
        if ($parentID === null) {
            $parentID = '';
        }

        $boardIDs = [];

        if (isset($this->cachedBoardStructure[$parentID])) {
            $boardIDs = $this->cachedBoardStructure[$parentID];
            foreach ($this->cachedBoardStructure[$parentID] as $boardID) {
                $boardIDs = \array_merge($boardIDs, $this->getAllChildIDs($boardID));
            }
        }

        return $boardIDs;
    }

    /**
     * Returns the number of clicks.
     *
     * @param   int     $boardID
     * @return  int
     */
    public function getClicks($boardID)
    {
        if (isset($this->counts[$boardID])) {
            return $this->counts[$boardID]['clicks'];
        }

        return 0;
    }

    /**
     * Returns the number of threads.
     *
     * @param   int     $boardID
     * @return  int
     */
    public function getThreads($boardID)
    {
        if (isset($this->counts[$boardID])) {
            return $this->counts[$boardID]['threads'];
        }

        return 0;
    }

    /**
     * Returns the number of posts.
     *
     * @param   int     $boardID
     * @return  int
     */
    public function getPosts($boardID)
    {
        if (isset($this->counts[$boardID])) {
            return $this->counts[$boardID]['posts'];
        }

        return 0;
    }

    /**
     * Returns the last post.
     *
     * @param   int     $boardID
     * @param   int     $languageID
     * @return  null|ViewableThread
     */
    public function getLastPost($boardID, $languageID = null)
    {
        if ($this->lastPosts === null) {
            $this->initLastPosts();
        }

        return $this->lastPosts[$boardID][$languageID] ?? null;
    }

    /**
     * Loads the last posts.
     */
    protected function initLastPosts()
    {
        $this->lastPosts = [];

        // handle private boards
        $privateBoardIDs = $privateThreadIDs = [];
        if (WCF::getUser()->userID) {
            foreach ($this->getBoards() as $board) {
                if ($board->isPrivate && !$board->canReadPrivateThreads()) {
                    $privateBoardIDs[] = $board->boardID;
                    $privateThreadIDs[$board->boardID] = 0;
                }
            }

            if (!empty($privateBoardIDs)) {
                $innerConditions = new PreparedStatementConditionBuilder();
                $innerConditions->add("boardID = board.boardID");
                $innerConditions->add("userID = ?", [WCF::getUser()->userID]);
                $innerConditions->add("isDeleted = ?", [0]);
                $innerConditions->add("isDisabled = ?", [0]);
                $innerConditions->add("movedThreadID IS NULL");

                $languageIDs = LanguageFactory::getInstance()->getContentLanguageIDs();
                if (!empty($languageIDs)) {
                    $innerConditions->add("languageID IN (?)", [$languageIDs]);
                }

                $outerConditions = new PreparedStatementConditionBuilder();
                $outerConditions->add("board.boardID IN (?)", [$privateBoardIDs]);

                $sql = "SELECT  board.boardID,
                                (
                                    SELECT      threadID
                                    FROM        wbb" . WCF_N . "_thread
                                    " . $innerConditions . "
                                    ORDER BY    lastPostTime DESC
                                    LIMIT       1
                                ) AS threadID
                        FROM    wbb" . WCF_N . "_board board
                        " . $outerConditions;
                $statement = WCF::getDB()->prepareStatement($sql);
                $statement->execute(
                    \array_merge($innerConditions->getParameters(), $outerConditions->getParameters())
                );
                while ($row = $statement->fetchArray()) {
                    if (!$row['threadID']) {
                        continue;
                    }

                    $this->lastPostThreadIDs[] = $row['threadID'];
                    $privateThreadIDs[$row['boardID']] = $row['threadID'];
                }
            }
        }

        if (!empty($this->lastPostThreadIDs)) {
            $threadList = new ViewableThreadList(false, true);
            $threadList->setObjectIDs($this->lastPostThreadIDs);
            $threadList->sqlOrderBy = 'lastPostTime DESC, lastPostID DESC';
            $threadList->readObjects();

            foreach ($threadList as $thread) {
                if (
                    isset($privateThreadIDs[$thread->boardID])
                    && $privateThreadIDs[$thread->boardID] != $thread->threadID
                ) {
                    // ignore threads for private boards that have been loaded
                    // using to the global last post cache
                    continue;
                }

                $this->lastPosts[$thread->boardID][$thread->languageID] = $thread;
            }
        }
    }

    /**
     * Returns true if the requested board is opened.
     *
     * @param   int     $boardID
     * @return  bool
     */
    public function isOpen($boardID)
    {
        if ($this->closedBoardIDs === null) {
            $this->closedBoardIDs = UserCollapsibleContentHandler::getInstance()->getCollapsedContent(
                UserCollapsibleContentHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board')
            );
        }

        return !\in_array($boardID, $this->closedBoardIDs);
    }

    /**
     * Returns true if the user has ignored the given board.
     *
     * @param   int     $boardID
     * @return  bool
     * @deprecated 5.5 Use `BoardStatusHandler::getInstance()->getSubscriptionStatus()` instead.
     */
    public function isIgnored($boardID)
    {
        return BoardStatusHandler::getInstance()->getSubscriptionStatus($boardID) === BoardStatusHandler::SUBSCRIPTION_MODE_IGNORING;
    }

    /**
     * Returns the number of unread threads.
     *
     * @param   int     $boardID
     * @return  int
     */
    public function getUnreadThreads($boardID)
    {
        if ($this->unreadThreads === null) {
            $this->initUnreadThreads();
        }

        if (isset($this->unreadThreads[$boardID])) {
            return $this->unreadThreads[$boardID];
        }

        return 0;
    }

    /**
     * Returns a list of all boards.
     *
     * @return  Board[]
     */
    public function getBoards()
    {
        return $this->cachedBoards;
    }

    /**
     * Returns the ids of the label groups associated with the board with the
     * given id. If no board is given, the complete mapping data is returned.
     *
     * @param   int     $boardID
     * @return  array
     */
    public function getLabelGroupIDs($boardID = null)
    {
        if ($boardID === null) {
            return $this->cachedLabelGroups;
        }

        if (isset($this->cachedLabelGroups[$boardID])) {
            return $this->cachedLabelGroups[$boardID];
        }

        return [];
    }

    /**
     * Returns the users online list.
     *
     * @param   int     $boardID
     * @return  UserOnline[]
     */
    public function getUsersOnline($boardID)
    {
        if ($this->usersOnline === null) {
            $this->initUsersOnline();
        }

        if (isset($this->usersOnline[$boardID])) {
            return $this->usersOnline[$boardID];
        }

        return [];
    }

    /**
     * Returns a list of user moderators for given board id.
     *
     * @param   int     $boardID
     * @return  int[]
     */
    public function getUserModerators($boardID)
    {
        if (isset($this->cachedModerators['users'][$boardID])) {
            return $this->cachedModerators['users'][$boardID];
        }

        return [];
    }

    /**
     * Returns a list of group moderators for given board id.
     *
     * @param   int     $boardID
     * @return  int[]
     */
    public function getGroupModerators($boardID)
    {
        if (isset($this->cachedModerators['groups'][$boardID])) {
            return $this->cachedModerators['groups'][$boardID];
        }

        return [];
    }
}
