<?php

namespace wbb\acp\form;

use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\board\RealtimeBoardNodeList;
use wbb\data\rss\feed\RssFeedAction;
use wbb\system\label\object\RssFeedLabelObjectHandler;
use wcf\data\label\group\ViewableLabelGroup;
use wcf\data\user\User;
use wcf\form\AbstractForm;
use wcf\system\exception\UserInputException;
use wcf\system\language\LanguageFactory;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;
use wcf\util\ArrayUtil;
use wcf\util\StringUtil;

/**
 * Shows the RSS feed add form.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Acp\Form
 */
class RssFeedAddForm extends AbstractForm
{
    /**
     * @inheritDoc
     */
    public $activeMenuItem = 'wbb.acp.menu.link.rssFeed.add';

    /**
     * @inheritDoc
     */
    public $neededPermissions = ['admin.board.canManageRssFeed'];

    /**
     * board node list
     * @var RealtimeBoardNodeList
     */
    public $boardNodeList;

    /**
     * feed title
     * @var string
     */
    public $title = '';

    /**
     * feed URL
     * @var string
     */
    public $url = '';

    /**
     * cycle time
     * @var int
     */
    public $cycleTime = 1800;

    /**
     * max items to fetch
     * @var int
     */
    public $maxResults = 0;

    /**
     * search keywords
     * @var string
     */
    public $searchKeywords = '';

    /**
     * board id
     * @var int
     */
    public $boardID = 0;

    /**
     * username
     * @var string
     */
    public $username = '';

    /**
     * user object
     * @var User
     */
    public $user;

    /**
     * language id
     * @var int
     */
    public $languageID = 0;

    /**
     * close thread after creation
     * @var int
     */
    public $closeThread = 0;

    /**
     * disable thread after creation
     * @var int
     */
    public $disableThread = 0;

    /**
     * threads tags
     * @var string[]
     */
    public $threadTags = [];

    /**
     * if 1, item categories are used as tags
     * @var int
     */
    public $useCategoriesAsTags = 0;

    /**
     * list of selected label ids
     * @var int[]
     */
    public $labelIDs = [];

    /**
     * list of available label groups for all boards
     * @var ViewableLabelGroup[]
     */
    public $labelGroups = [];

    /**
     * maps the label groups ids to the ids of the boards with whom they are
     * associated (labelGroupID => boardIDs)
     * @var array
     */
    public $labelGroupMapping = [];

    /**
     * maps the label groups ids to the ids of the boards with whom they are
     * associated (boardID => labelGroupIDs)
     * @var array
     */
    public $boardsToLabelGroups = [];

    /**
     * @inheritDoc
     */
    public function readFormParameters()
    {
        parent::readFormParameters();

        if (isset($_POST['title'])) {
            $this->title = StringUtil::trim($_POST['title']);
        }
        if (isset($_POST['url'])) {
            $this->url = StringUtil::trim($_POST['url']);
        }
        if (isset($_POST['cycleTime'])) {
            $this->cycleTime = \intval($_POST['cycleTime']);
        }
        if (isset($_POST['maxResults'])) {
            $this->maxResults = \intval($_POST['maxResults']);
        }
        if (isset($_POST['searchKeywords'])) {
            $this->searchKeywords = StringUtil::trim($_POST['searchKeywords']);
        }
        if (isset($_POST['boardID'])) {
            $this->boardID = \intval($_POST['boardID']);
        }
        if (isset($_POST['labelIDs']) && \is_array($_POST['labelIDs'])) {
            $this->labelIDs = $_POST['labelIDs'];
        } else {
            $this->labelIDs = [];
        }
        if (isset($_POST['username'])) {
            $this->username = StringUtil::trim($_POST['username']);
        }
        if (isset($_POST['languageID']) && LanguageFactory::getInstance()->multilingualismEnabled()) {
            $this->languageID = \intval($_POST['languageID']);
        }
        if (!empty($_POST['closeThread'])) {
            $this->closeThread = 1;
        }
        if (!empty($_POST['disableThread'])) {
            $this->disableThread = 1;
        }
        if (!empty($_POST['useCategoriesAsTags'])) {
            $this->useCategoriesAsTags = 1;
        }
        if (
            MODULE_TAGGING
            && WBB_THREAD_ENABLE_TAGS
            && isset($_POST['threadTags'])
            && \is_array($_POST['threadTags'])
        ) {
            $this->threadTags = ArrayUtil::trim($_POST['threadTags']);
        }
    }

    /**
     * @inheritDoc
     */
    public function validate()
    {
        parent::validate();

        // title
        if (empty($this->title)) {
            throw new UserInputException('title');
        }

        // feed url
        if (empty($this->url)) {
            throw new UserInputException('url');
        }

        // board id
        $board = new Board($this->boardID);
        if (!$board->boardID || !$board->isBoard()) {
            throw new UserInputException('boardID');
        }

        // label ids
        if (!empty($this->labelGroups)) {
            $validLabelGroupIDs = BoardCache::getInstance()->getLabelGroupIDs($this->boardID);

            // discard label ids of other groups
            foreach ($this->labelIDs as $groupID => $labelID) {
                if (!\in_array($groupID, $validLabelGroupIDs)) {
                    unset($this->labelIDs[$groupID]);
                }
            }

            RssFeedLabelObjectHandler::getInstance()->setBoardID($this->boardID);
            $validationResult = RssFeedLabelObjectHandler::getInstance()->validateLabelIDs(
                $this->labelIDs,
                'canSetLabel',
                false
            );
            if (!empty($validationResult[0])) {
                throw new UserInputException('labelIDs');
            }

            if (!empty($validationResult)) {
                throw new UserInputException('label', $validationResult);
            }
        } else {
            $this->labelIDs = [];
        }

        // user
        if (empty($this->username)) {
            throw new UserInputException('username');
        }
        $this->user = User::getUserByUsername($this->username);
        if (!$this->user->userID) {
            throw new UserInputException('username', 'notFound');
        }

        // language id
        if (LanguageFactory::getInstance()->multilingualismEnabled()) {
            if (LanguageFactory::getInstance()->getLanguage($this->languageID) === null) {
                throw new UserInputException('languageID');
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function save()
    {
        parent::save();

        $this->objectAction = new RssFeedAction([], 'create', [
            'data' => \array_merge($this->additionalFields, [
                'title' => $this->title,
                'url' => $this->url,
                'cycleTime' => $this->cycleTime,
                'maxResults' => $this->maxResults,
                'searchKeywords' => $this->searchKeywords,
                'boardID' => $this->boardID,
                'userID' => $this->user->userID,
                'languageID' => $this->languageID ?: null,
                'closeThread' => $this->closeThread,
                'disableThread' => $this->disableThread,
                'threadTags' => \implode(',', $this->threadTags),
                'useCategoriesAsTags' => $this->useCategoriesAsTags,
            ]),
            'labelIDs' => $this->labelIDs,
        ]);
        $this->objectAction->executeAction();

        // reset values
        $this->title = $this->url = $this->searchKeywords = $this->username = '';
        $this->maxResults = $this->boardID = $this->languageID = $this->useCategoriesAsTags = $this->closeThread = $this->disableThread = 0;
        $this->cycleTime = 1800;
        $this->labelIDs = $this->threadTags = [];

        // show success message
        WCF::getTPL()->assign([
            'success' => true,
            'objectEditLink' => LinkHandler::getInstance()->getControllerLink(
                RssFeedEditForm::class,
                ['id' => $this->objectAction->getReturnValues()['returnValues']->feedID]
            ),
        ]);
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        $this->labelGroups = RssFeedLabelObjectHandler::getInstance()->getLabelGroups();
        $this->boardsToLabelGroups = BoardCache::getInstance()->getLabelGroupIDs();
        foreach ($this->boardsToLabelGroups as $boardID => $labelGroupIDs) {
            foreach ($labelGroupIDs as $labelGroupID) {
                if (!isset($this->labelGroupMapping[$labelGroupID])) {
                    $this->labelGroupMapping[$labelGroupID] = [];
                }
                $this->labelGroupMapping[$labelGroupID][] = $boardID;
            }
        }
        foreach ($this->labelGroups as $labelGroup) {
            if (!isset($this->labelGroupMapping[$labelGroup->groupID])) {
                unset($this->labelGroups[$labelGroup->groupID]);
            }
        }

        parent::readData();

        // get board list
        $this->boardNodeList = new RealtimeBoardNodeList();
        $this->boardNodeList->readNodeTree();
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        WCF::getTPL()->assign([
            'action' => 'add',
            'title' => $this->title,
            'url' => $this->url,
            'cycleTime' => $this->cycleTime,
            'maxResults' => $this->maxResults,
            'searchKeywords' => $this->searchKeywords,
            'boardID' => $this->boardID,
            'username' => $this->username,
            'languageID' => $this->languageID,
            'closeThread' => $this->closeThread,
            'disableThread' => $this->disableThread,
            'threadTags' => $this->threadTags,
            'useCategoriesAsTags' => $this->useCategoriesAsTags,
            'boardNodeList' => $this->boardNodeList->getNodeList(),
            'multilingualismEnabled' => LanguageFactory::getInstance()->multilingualismEnabled(),
            'availableLanguages' => LanguageFactory::getInstance()->getContentLanguages(),
            'labelGroups' => $this->labelGroups,
            'labelGroupMapping' => $this->labelGroupMapping,
            'boardsToLabelGroups' => $this->boardsToLabelGroups,
            'labelIDs' => $this->labelIDs,
        ]);
    }
}
