/**
 * Handles the mark as read button for individual threads.
 *
 * @author Alexander Ebert, Joshua Ruesweg
 * @copyright 2001-2022 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module WoltLabSuite/Forum/Ui/Thread/LastPageHandler
 * @since 5.5
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Ajax", "WoltLabSuite/Core/Language", "WoltLabSuite/Core/Dom/Util", "WoltLabSuite/Core/StringUtil", "WoltLabSuite/Core/Core", "WoltLabSuite/Core/Dom/Change/Listener", "WoltLabSuite/Core/Notification/Handler"], function (require, exports, tslib_1, Ajax_1, Language, DomUtil, StringUtil_1, Core_1, Listener_1, Handler_1) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.setup = exports.ready = exports.checkForNewPosts = void 0;
    Language = tslib_1.__importStar(Language);
    DomUtil = tslib_1.__importStar(DomUtil);
    Listener_1 = tslib_1.__importDefault(Listener_1);
    let callbackIsReady;
    let isIdle = true;
    let isEnabled;
    let messageQuickReply = null;
    let notice;
    let promiseIsReady;
    let thread;
    let threadPostList;
    function getLastContentElement() {
        const messageQuickReply = document.getElementById("messageQuickReply");
        if (messageQuickReply !== null) {
            const previousElement = messageQuickReply.previousElementSibling;
            if (previousElement.classList.contains("messageListPagination")) {
                return previousElement.previousElementSibling;
            }
            else {
                return previousElement;
            }
        }
        else {
            const elements = Array.from(document.querySelectorAll(".wbbThreadPostList .wbbPost"));
            const lastPost = elements.pop();
            return lastPost.closest("li");
        }
    }
    function showNewPostNotice(newPostsCount) {
        if (notice === undefined) {
            notice = document.createElement("li");
            notice.classList.add("wbbMorePostsNotice");
            notice.innerHTML = '<a href="#" id="wbbNewPostsLink" class="button buttonPrimary"></a>';
            notice.addEventListener("click", (event) => {
                event.preventDefault();
                void loadNewPosts();
            });
        }
        const button = notice.querySelector(".button");
        button.innerHTML = Language.get("wbb.thread.newPosts", { newPostsCount });
        getLastContentElement().insertAdjacentElement("afterend", notice);
    }
    async function loadNewPosts() {
        const result = (await (0, Ajax_1.dboAction)("getNewPosts", "wbb\\data\\thread\\ThreadAction")
            .objectIds([parseInt(thread.dataset.threadId || "", 10)])
            .payload({
            countPosts: false,
            lastPostTime: parseInt(threadPostList.dataset.lastPostTime || "", 10),
            pageNo: parseInt(threadPostList.dataset.pageNo || "", 10),
        })
            .dispatch());
        if ("redirectURL" in result) {
            window.location.href = result.redirectURL;
        }
        else {
            notice.remove();
            DomUtil.insertHtml(result.template, getLastContentElement(), "after");
            if (result.nextPageNotice) {
                const html = `<li class="wbbMorePostsNotice"><a href="${(0, StringUtil_1.escapeHTML)(result.nextPageUrl)}" class="button buttonPrimary">${result.nextPageNotice}</a></li>`;
                DomUtil.insertHtml(html, getLastContentElement(), "after");
            }
            threadPostList.dataset.lastPostTime = result.lastPostTime.toString();
            if (messageQuickReply) {
                messageQuickReply.dataset.lastPostTime = result.lastPostTime.toString();
            }
            Listener_1.default.trigger();
            // Loading new posts implicitly marks the notifications for these posts
            // as read. This can cause the UI to be out of sync when the notifications
            // are accessed before the new post is loaded.
            (0, Handler_1.poll)();
        }
    }
    /**
     * Observes the scroll position and triggers a check for new posts
     * once the last post becomes visible.
     */
    function observeScrollPosition() {
        const post = document.querySelector(".wbbThreadPostList .wbbPost");
        if (post === null) {
            return;
        }
        const observer = new IntersectionObserver((entries, observer) => {
            if (entries.some((entry) => entry.isIntersecting)) {
                observer.unobserve(post);
                void checkForNewPosts();
            }
        });
        observer.observe(post);
    }
    async function checkForNewPosts() {
        if (!isIdle) {
            return;
        }
        isIdle = false;
        let result;
        try {
            result = (await (0, Ajax_1.dboAction)("getNewPosts", "wbb\\data\\thread\\ThreadAction")
                .objectIds([parseInt(thread.dataset.threadId || "", 10)])
                .payload({
                countPosts: true,
                lastPostTime: parseInt(threadPostList.dataset.lastPostTime || "", 10),
            })
                .disableLoadingIndicator()
                .dispatch());
        }
        finally {
            isIdle = true;
        }
        if (result.newPostsCount > 0) {
            showNewPostNotice(result.newPostsCount);
        }
    }
    exports.checkForNewPosts = checkForNewPosts;
    function ready() {
        if (promiseIsReady === undefined) {
            promiseIsReady = new Promise((resolve) => {
                callbackIsReady = () => resolve(isEnabled);
            });
            if (isEnabled !== undefined) {
                callbackIsReady();
            }
        }
        return promiseIsReady;
    }
    exports.ready = ready;
    function setup() {
        thread = document.querySelector(".wbbThread");
        threadPostList = document.querySelector(".wbbThreadPostList");
        messageQuickReply = document.getElementById("messageQuickReply");
        if ((0, Core_1.stringToBool)(threadPostList.dataset.isLastPage || "") && threadPostList.dataset.sortOrder === "ASC") {
            isEnabled = true;
            observeScrollPosition();
            if (messageQuickReply) {
                // Replying to a thread will update the timestamp of the quick reply element,
                // but is unaware of the post list. This observer monitors changes to the
                // last post time and mirrors it to the post list to keep those values in sync.
                const observer = new MutationObserver(() => {
                    const threadLastPostTime = parseInt(threadPostList.dataset.lastPostTime || "", 10);
                    const replyLastPostTime = parseInt(messageQuickReply.dataset.lastPostTime || "", 10);
                    if (threadLastPostTime < replyLastPostTime) {
                        threadPostList.dataset.lastPostTime = replyLastPostTime.toString();
                        // Replying to the thread will implicitly load all pending posts,
                        // therefore we can safely remove the notice once this happens.
                        notice === null || notice === void 0 ? void 0 : notice.remove();
                    }
                });
                observer.observe(messageQuickReply, {
                    attributes: true,
                    attributeFilter: ["data-last-post-time"],
                });
            }
        }
        else {
            isEnabled = false;
        }
        if (callbackIsReady !== undefined) {
            callbackIsReady();
        }
    }
    exports.setup = setup;
});
