<?php

namespace wbb\system\event\listener;

use wbb\data\board\BoardCache;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\event\listener\IParameterizedEventListener;
use wcf\system\user\activity\point\UserActivityPointHandler;
use wcf\system\WCF;
use wcf\system\worker\UserActivityPointItemsRebuildDataWorker;

/**
 * Updates the user activity point items counter for forum threads and posts.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Event\Listener
 * @since   5.2
 */
class UserActivityPointItemsRebuildDataWorkerListener implements IParameterizedEventListener
{
    /**
     * @inheritDoc
     */
    public function execute($eventObj, $className, $eventName, array &$parameters)
    {
        /** @var UserActivityPointItemsRebuildDataWorker $eventObj */

        $threadObjectType = UserActivityPointHandler::getInstance()
            ->getObjectTypeByName('com.woltlab.wbb.activityPointEvent.thread');

        $conditionBuilder = new PreparedStatementConditionBuilder();
        $conditionBuilder->add('user_activity_point.objectTypeID = ?', [$threadObjectType->objectTypeID]);
        $conditionBuilder->add('user_activity_point.userID IN (?)', [$eventObj->getObjectList()->getObjectIDs()]);

        $boardIDs = [];
        foreach (BoardCache::getInstance()->getBoards() as $board) {
            if ($board->countUserPosts) {
                $boardIDs[] = $board->boardID;
            }
        }

        if (!empty($boardIDs)) {
            $subConditionBuilder = new PreparedStatementConditionBuilder();
            $subConditionBuilder->add('thread.userID = user_table.userID');
            $subConditionBuilder->add('thread.boardID IN (?)', [$boardIDs]);
            $subConditionBuilder->add('thread.isDisabled = ?', [0]);

            $sql = "UPDATE      wcf" . WCF_N . "_user_activity_point user_activity_point
                    LEFT JOIN   wcf" . WCF_N . "_user user_table
                    ON          user_table.userID = user_activity_point.userID
                    SET         user_activity_point.items = (
                                    SELECT  COUNT(*)
                                    FROM    wbb" . WCF_N . "_thread thread
                                    " . $subConditionBuilder . "
                                ),
                                user_activity_point.activityPoints = user_activity_point.items * ?
                    " . $conditionBuilder;
            $statement = WCF::getDB()->prepareStatement($sql);
            $statement->execute(\array_merge(
                $subConditionBuilder->getParameters(),
                [$threadObjectType->points],
                $conditionBuilder->getParameters()
            ));
        }

        $postObjectType = UserActivityPointHandler::getInstance()
            ->getObjectTypeByName('com.woltlab.wbb.activityPointEvent.post');

        $conditionBuilder = new PreparedStatementConditionBuilder();
        $conditionBuilder->add('user_activity_point.objectTypeID = ?', [$postObjectType->objectTypeID]);
        $conditionBuilder->add('user_activity_point.userID IN (?)', [$eventObj->getObjectList()->getObjectIDs()]);

        // `wbbPosts` also includes initial posts of threads, thus subtract the previously calculated
        // number of threads
        $sql = "UPDATE      wcf" . WCF_N . "_user_activity_point user_activity_point
                LEFT JOIN   wcf" . WCF_N . "_user user_table
                ON          user_table.userID = user_activity_point.userID
                LEFT JOIN   wcf" . WCF_N . "_user_activity_point user_activity_point_thread
                ON          user_activity_point_thread.userID = user_activity_point.userID
                        AND user_activity_point_thread.objectTypeID = ?
                SET         user_activity_point.items = user_table.wbbPosts - COALESCE(user_activity_point_thread.items, 0),
                            user_activity_point.activityPoints = user_activity_point.items * ?
                " . $conditionBuilder;
        $statement = WCF::getDB()->prepareStatement($sql);
        $statement->execute(\array_merge(
            [$threadObjectType->objectTypeID, $postObjectType->points],
            $conditionBuilder->getParameters()
        ));
    }
}
