<?php

namespace wbb\data\board;

/**
 * Represents a board node tree.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Board
 */
class BoardNodeTree
{
    /**
     * parent board id
     * @var int
     */
    public $parentID;

    /**
     * start depth
     * @var int
     */
    public $startDepth = 0;

    /**
     * list of board ids
     * @var int[]
     */
    public $filter = [];

    /**
     * root node
     * @var BoardNode
     */
    public $node;

    /**
     * board node class name
     * @var string
     */
    public $nodeClassname = BoardNode::class;

    /**
     * Creates a new BoardNodeTree object.
     *
     * @param   int     $parentID
     * @param   int     $startDepth
     * @param   int[]   $filter
     */
    public function __construct($parentID = null, $startDepth = 0, array $filter = [])
    {
        $this->parentID = $parentID;
        $this->startDepth = $startDepth;
        $this->filter = $filter;
    }

    /**
     * Reads the node tree.
     */
    public function readNodeTree()
    {
        // generate node tree
        $this->node = new BoardNode(null, null, $this->startDepth);
        $this->node->setChildren($this->generateNodeTree($this->parentID, $this->node, $this->filter));
    }

    /**
     * Generates the node tree recursively
     *
     * @param   int     $parentID
     * @param   BoardNode   $parentNode
     * @param   int[]   $filter
     * @return  BoardNode[]
     */
    protected function generateNodeTree($parentID, ?BoardNode $parentNode = null, array $filter = [])
    {
        $nodes = [];

        // apply filter
        $boardIDs = BoardCache::getInstance()->getChildIDs($parentID);
        if (\count($filter)) {
            $boardIDs = \array_intersect($boardIDs, $filter);
        }

        foreach ($boardIDs as $boardID) {
            $board = BoardCache::getInstance()->getBoard($boardID);
            if ($this->isVisible($board)) {
                /** @var BoardNode $node */
                $node = new $this->nodeClassname(
                    $parentNode,
                    $board,
                    $parentNode !== null ? $parentNode->getDepth() + 1 : 0
                );
                $nodes[] = $node;

                // get children
                $node->setChildren($this->generateNodeTree($boardID, $node));
            }
        }

        return $nodes;
    }

    /**
     * Returns the board node tree.
     *
     * @return  BoardNode[]
     */
    public function getNodeTree()
    {
        return $this->node->getChildren();
    }

    /**
     * Returns true if the given board is visible.
     *
     * @param   Board   $board
     * @return  bool
     */
    protected function isVisible(Board $board)
    {
        return true;
    }

    /**
     * Filters the list of nodes using a given callback function.
     *
     * @since 5.5
     */
    public function filter(callable $filter): void
    {
        $this->node->filter($filter);
    }
}
