<?php

namespace wbb\data\board;

use wbb\data\thread\ViewableThread;
use wcf\data\user\online\UserOnline;
use wcf\system\language\LanguageFactory;
use wcf\system\WCF;

/**
 * Represents a board node element.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Board
 */
class DetailedBoardNode extends BoardNode
{
    /**
     * number of threads in this board
     * @var int
     */
    protected $threads;

    /**
     * number of posts in this board
     * @var int
     */
    protected $posts;

    /**
     * number of unread threads in this board
     * @var int
     */
    protected $unreadThreads;

    /**
     * number of unread threads in this board (including all subboards)
     * @var int
     */
    protected $totalUnreadThreads;

    /**
     * last post in this board
     * @var \wbb\data\thread\Thread
     */
    protected $lastPost;

    /**
     * true if the last post was loaded already
     * @var bool
     */
    protected $lastPostLoaded = false;

    /**
     * list of sub boards
     * @var BoardNode[]
     */
    protected $subBoards;

    /**
     * Returns the number of clicks.
     *
     * @return  int
     */
    public function getClicks()
    {
        return BoardCache::getInstance()->getClicks($this->board->boardID);
    }

    /**
     * Returns the number of threads.
     *
     * @return  int
     */
    public function getThreads()
    {
        if ($this->threads === null) {
            $this->threads = $this->board->getThreads();
            if (!$this->hasVisibleChildren()) {
                foreach ($this->children as $childNode) {
                    $this->threads += $childNode->getThreads();
                }
            }
        }

        return $this->threads;
    }

    /**
     * Returns the number of posts.
     *
     * @return  int
     */
    public function getPosts()
    {
        if ($this->posts === null) {
            $this->posts = $this->board->getPosts();
            if (!$this->hasVisibleChildren()) {
                foreach ($this->children as $childNode) {
                    $this->posts += $childNode->getPosts();
                }
            }
        }

        return $this->posts;
    }

    /**
     * Returns the number of unread threads in this board.
     *
     * @return  int
     */
    public function getUnreadThreads()
    {
        if ($this->unreadThreads === null) {
            $this->unreadThreads = BoardCache::getInstance()->getUnreadThreads($this->board->boardID);
            if (
                !$this->hasVisibleChildren()
                && (
                    $this->board->isCategory()
                    || !WBB_BOARD_LIST_ENABLE_SUB_BOARDS
                    || !$this->board->showSubBoards
                    || ($this->parentNode !== null && !$this->getParentNode()->isOpenNode())
                )
            ) {
                foreach ($this->children as $childNode) {
                    $this->unreadThreads += $childNode->getUnreadThreads();
                }
            }
        }

        return $this->unreadThreads;
    }

    /**
     * Returns the number of unread threads in this board (including all subboards).
     *
     * @return  int
     */
    public function getTotalUnreadThreads()
    {
        if ($this->totalUnreadThreads === null) {
            $this->totalUnreadThreads = BoardCache::getInstance()->getUnreadThreads($this->board->boardID);
            foreach ($this->children as $childNode) {
                $this->totalUnreadThreads += $childNode->getUnreadThreads();
            }
        }

        return $this->totalUnreadThreads;
    }

    /**
     * Returns the last post.
     *
     * @return  ViewableThread
     */
    public function getLastPost()
    {
        if (!$this->lastPostLoaded) {
            $this->lastPostLoaded = true;
            if ($this->board->getPermission('canEnterBoard')) {
                $lastPosts = [];

                if (
                    LanguageFactory::getInstance()->multilingualismEnabled()
                    && \count(WCF::getUser()->getLanguageIDs())
                ) {
                    foreach (WCF::getUser()->getLanguageIDs() as $languageID) {
                        $lastPost = BoardCache::getInstance()->getLastPost($this->board->boardID, $languageID);
                        if ($lastPost !== null) {
                            $lastPosts[] = $lastPost;
                        }
                    }
                } else {
                    $lastPost = BoardCache::getInstance()->getLastPost($this->board->boardID);
                    if ($lastPost !== null) {
                        $lastPosts[] = $lastPost;
                    }
                }

                if (!$this->hasVisibleChildren()) {
                    foreach ($this->children as $childNode) {
                        if (($childLastPost = $childNode->getLastPost()) !== null) {
                            $lastPosts[] = $childLastPost;
                        }
                    }
                }

                \usort($lastPosts, static function (ViewableThread $lastPostA, ViewableThread $lastPostB) {
                    return $lastPostB->lastPostTime <=> $lastPostA->lastPostTime
                        ?: $lastPostB->threadID <=> $lastPostA->threadID;
                });
                if ($lastPosts !== []) {
                    $this->lastPost = \reset($lastPosts);
                }
            }
        }

        return $this->lastPost;
    }

    /**
     * Returns the sub boards.
     *
     * @return  array
     */
    public function getSubBoards()
    {
        if ($this->subBoards === null) {
            $this->subBoards = [];
            if (WBB_BOARD_LIST_ENABLE_SUB_BOARDS && $this->board->showSubBoards && !$this->hasVisibleChildren()) {
                foreach ($this->children as $childNode) {
                    $this->subBoards[] = $childNode;
                    $childSubBoards = $childNode->getSubBoards();
                    if (\count($childSubBoards)) {
                        $this->subBoards = \array_merge($this->subBoards, $childSubBoards);
                    }
                }
            }
        }

        return $this->subBoards;
    }

    /**
     * Returns sorted sub boards.
     *
     * @return  array
     */
    public function getSortedSubBoards()
    {
        $subBoards = $this->getSubBoards();
        \usort($subBoards, static function (BoardNode $boardNodeA, BoardNode $boardNodeB) {
            return \strcoll($boardNodeA->getBoard()->title, $boardNodeB->getBoard()->title);
        });

        return $subBoards;
    }

    /**
     * Returns true if a collapsible category is open.
     *
     * @return  bool
     */
    public function isOpen()
    {
        if ($this->board !== null && $this->board->isCategory()) {
            return BoardCache::getInstance()->isOpen($this->board->boardID);
        }

        return true;
    }

    /**
     * Returns true if this node is open.
     *
     * @return  bool
     */
    public function isOpenNode()
    {
        return $this->isOpen() && $this->depth + 1 <= WBB_BOARD_LIST_DEPTH;
    }

    /**
     * Returns true if this node has visible children.
     *
     * @return  bool
     */
    public function hasVisibleChildren()
    {
        return $this->hasChildren() && $this->isOpenNode();
    }

    /**
     * Returns true if this node is collapsible.
     *
     * @return  bool
     */
    public function isCollapsible()
    {
        if (
            $this->board !== null
            && $this->board->isCategory()
            && $this->hasChildren()
            && $this->depth + 1 <= WBB_BOARD_LIST_DEPTH
        ) {
            return true;
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    public function valid()
    {
        return parent::valid() && $this->isOpenNode();
    }

    /**
     * Returns the users online list.
     *
     * @return  UserOnline[]
     */
    public function getUsersOnline()
    {
        return BoardCache::getInstance()->getUsersOnline($this->board->boardID);
    }

    /**
     * Returns the parent node
     *
     * @return  DetailedBoardNode|null
     */
    public function getParentNode()
    {
        if ($this->parentNode === null) {
            return null;
        }

        return new self($this->parentNode->parentNode, $this->parentNode->getBoard(), $this->depth - 1);
    }
}
