<?php

namespace wbb\system\condition;

use wbb\data\board\BoardCache;
use wbb\data\board\BoardNodeList;
use wbb\page\IBoardPage;
use wcf\data\condition\Condition;
use wcf\data\page\PageCache;
use wcf\system\condition\AbstractMultiSelectCondition;
use wcf\system\condition\IContentCondition;
use wcf\system\request\RequestHandler;
use wcf\util\StringUtil;

/**
 * Condition implementation for selecting multiple boards.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Condition
 */
class PageBoardCondition extends AbstractMultiSelectCondition implements IContentCondition
{
    /**
     * @inheritDoc
     */
    protected $description = 'wbb.condition.page.board.description';

    /**
     * @inheritDoc
     */
    protected $fieldName = 'wbbPageBoardIDs';

    /**
     * @inheritDoc
     */
    protected $label = 'wbb.condition.page.board';

    /**
     * @inheritDoc
     */
    protected function getFieldElement()
    {
        $boardNodeList = new BoardNodeList();
        $boardNodeList->readNodeTree();
        $boardNodeList = $boardNodeList->getNodeList();
        $boardCount = \count(BoardCache::getInstance()->getBoards());

        $fieldElement = '<select name="' . $this->fieldName . '[]" id="' . $this->fieldName . '" multiple size="' . ($boardCount > 10 ? 10 : $boardCount) . '">';
        foreach ($boardNodeList as $board) {
            $fieldElement .= '<option value="' . $board->getBoard()->boardID . '"' . (\in_array($board->getBoard()->boardID, $this->fieldValue) ? ' selected' : '') . '>' . \str_repeat('&nbsp;&nbsp;&nbsp;&nbsp;', $board->getDepth() - 1) . StringUtil::encodeHTML($board->getBoard()->getTitle()) . '</option>';
        }
        $fieldElement .= "</select>";

        $pageIDs = [];
        foreach (PageCache::getInstance()->getPages() as $page) {
            if (\is_subclass_of($page->controller, IBoardPage::class)) {
                $pageIDs[] = $page->pageID;
            }
        }
        $pageIDsString = \implode(', ', $pageIDs);

        return <<<HTML
{$fieldElement}
<script>
	require(['WoltLabSuite/Core/Controller/Condition/Page/Dependence'], function(ControllerConditionPageDependence) {
		ControllerConditionPageDependence.register(elById('wbbPageBoardIDs').parentNode.parentNode, [ {$pageIDsString} ]);
	});
</script>
HTML;
    }

    /**
     * @inheritDoc
     */
    protected function getOptions()
    {
        return BoardCache::getInstance()->getBoards();
    }

    /**
     * @inheritDoc
     */
    public function showContent(Condition $condition)
    {
        if (RequestHandler::getInstance()->getActiveRequest() === null) {
            return false;
        }

        $requestObject = RequestHandler::getInstance()->getActiveRequest()->getRequestObject();
        if (!($requestObject instanceof IBoardPage)) {
            return false;
        }

        /** @noinspection PhpUndefinedFieldInspection */
        return $requestObject->getBoard() !== null && \in_array($requestObject->getBoard()->boardID, $condition->wbbPageBoardIDs);
    }
}
