/**
 * Handles toggling label groups based on a selected board.
 *
 * @author  Matthias Schmidt
 * @copyright  2001-2019 WoltLab GmbH
 * @license  WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module  WoltLabSuite/Forum/Controller/Thread/LabelCondition
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Core", "WoltLabSuite/Core/Dom/Util"], function (require, exports, tslib_1, Core, Util_1) {
    "use strict";
    Core = tslib_1.__importStar(Core);
    Util_1 = tslib_1.__importDefault(Util_1);
    class ControllerThreadLabelCondition {
        constructor(boardSelectorId, labelGroupMapping) {
            this._boardSelector = document.getElementById(boardSelectorId);
            if (this._boardSelector !== null) {
                this._boardSelector.addEventListener("change", () => this.change());
            }
            else {
                this._boardSelector = document.getElementById(`${boardSelectorId}Container`);
                if (this._boardSelector !== null) {
                    this._boardSelector
                        .querySelectorAll("input")
                        .forEach((el) => el.addEventListener("change", () => this.change()));
                }
                else {
                    throw new Error(`Unknown board selector with id '${boardSelectorId}.`);
                }
            }
            this._labelGroupMapping = labelGroupMapping;
            this._labelLists = Array.from(document.querySelectorAll(".labelList"));
            if (this._labelLists.length === 0) {
                return;
            }
            this.change();
            const form = this._labelLists[0].closest("form");
            form.addEventListener("submit", () => this.submit());
        }
        /**
         * Toggles the visible label groups based on the selected board.
         */
        change() {
            let boardIds;
            if (this._boardSelector instanceof HTMLSelectElement) {
                if (this._boardSelector.multiple) {
                    boardIds = Array.from(this._boardSelector.options)
                        .filter((option) => option.selected)
                        .map((option) => ~~option.value);
                }
                else {
                    boardIds = [~~this._boardSelector.value];
                }
            }
            else {
                boardIds = Array.from(this._boardSelector.querySelectorAll("input"))
                    .filter((option) => option.checked)
                    .map((option) => ~~option.value);
            }
            let labelGroupIds = [];
            if (boardIds.length) {
                labelGroupIds = this._labelGroupMapping[boardIds[0]] || [];
                labelGroupIds = labelGroupIds.filter((labelGroupID) => {
                    for (const boardId of boardIds) {
                        const localLabelGroupIDs = this._labelGroupMapping[boardId] || [];
                        if (!localLabelGroupIDs.includes(labelGroupID)) {
                            return false;
                        }
                    }
                    return true;
                });
            }
            this._labelLists.forEach((labelList) => {
                const objectId = ~~labelList.dataset.objectId;
                const dl = labelList.closest("dl");
                if (labelGroupIds.includes(objectId)) {
                    Util_1.default.show(dl);
                }
                else {
                    Util_1.default.hide(dl);
                }
            });
        }
        /**
         * Unsets selected labels for invisible label choosers.
         */
        submit() {
            this._labelLists.forEach((labelList) => {
                const dl = labelList.querySelector("dl");
                if (Util_1.default.isHidden(dl)) {
                    const labelChooser = labelList.querySelector(".labelChooser");
                    window.jQuery(labelChooser).data("labelID", 0);
                }
            });
        }
    }
    Core.enableLegacyInheritance(ControllerThreadLabelCondition);
    return ControllerThreadLabelCondition;
});
