<?php

namespace wbb\system\condition\thread;

use wbb\data\thread\ThreadList;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectList;
use wcf\system\condition\AbstractCheckboxCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\exception\InvalidObjectArgument;
use wcf\system\exception\SystemException;

/**
 * Condition implementation for threads to only include unread threads in lists.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Condition\Thread
 * @since   5.0
 */
class ThreadUnreadCondition extends AbstractCheckboxCondition implements IObjectListCondition
{
    /**
     * @inheritDoc
     */
    protected $fieldName = 'wbbThreadIsUnread';

    /**
     * @inheritDoc
     */
    protected $label = 'wbb.thread.condition.isUnread';

    /**
     * list of supported property names
     * @var string[]
     */
    protected $validPropertyNames = ['lastPostTime', 'time'];

    /**
     * @inheritDoc
     */
    public function __construct(DatabaseObject $object)
    {
        parent::__construct($object);

        /** @noinspection PhpUndefinedFieldInspection */
        $propertyName = $this->object->propertyName;

        if ($propertyName) {
            if (!\in_array($propertyName, $this->validPropertyNames)) {
                throw new SystemException("Unsupported property name '" . $propertyName . "'");
            }

            $this->fieldName .= \ucfirst($propertyName);
            $this->label .= '.' . $propertyName;
        }
    }

    /**
     * @inheritDoc
     */
    public function addObjectListCondition(DatabaseObjectList $objectList, array $conditionData)
    {
        if (!($objectList instanceof ThreadList)) {
            throw new InvalidObjectArgument($objectList, ThreadList::class, 'Object list');
        }

        if ($conditionData[$this->fieldName]) {
            /** @noinspection PhpUndefinedFieldInspection */
            $objectList->excludeReadThreads($this->object->propertyName ?: 'lastPostTime');
        }
    }
}
