/**
 * Basic update handler for threads used in thread view.
 *
 * @author  Joshua Ruesweg
 * @copyright  2001-2021 WoltLab GmbH
 * @license  WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module  WoltLabSuite/Forum/Handler/Thread/ThreadUpdateHandler
 * @since 5.5
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Dom/Util", "./UpdateHandler", "WoltLabSuite/Core/Language", "WoltLabSuite/Core/Timer/Repeating"], function (require, exports, tslib_1, Util_1, UpdateHandler_1, Language, Repeating_1) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.ThreadUpdateHandler = void 0;
    Util_1 = tslib_1.__importDefault(Util_1);
    UpdateHandler_1 = tslib_1.__importDefault(UpdateHandler_1);
    Language = tslib_1.__importStar(Language);
    Repeating_1 = tslib_1.__importDefault(Repeating_1);
    class ThreadUpdateHandler extends UpdateHandler_1.default {
        constructor(boardID) {
            super();
            this.boardID = 0;
            this.boardID = boardID;
        }
        setPostHandler(postHandler) {
            this.postHandler = postHandler;
        }
        update(threadId, data) {
            if (this.threads.get(threadId)) {
                if ("isDeleted" in data && !data.isDeleted && "ignoreDeletedPosts" in data) {
                    this.restore(threadId, true);
                    delete data.isDeleted;
                    delete data.ignoreDeletedPosts;
                }
                if ("isDisabled" in data && !data.isDisabled && "ignoreDisabledPosts" in data) {
                    this.enable(threadId, true);
                    delete data.isDisabled;
                    delete data.ignoreDisabledPosts;
                }
            }
            super.update(threadId, data, false);
        }
        close(threadId) {
            const metaData = this.getThread(threadId).querySelector(".contentHeaderMetaData");
            metaData.appendChild(getCloseElement());
            // Create warning for locked thread.
            const redactorBox = document.querySelector(".messageQuickReplyContent .redactor-box");
            if (redactorBox) {
                const warning = redactorBox.closest(".jsClosedThreadWarning");
                if (!warning) {
                    const warningElement = document.createElement("p");
                    warningElement.classList.add("warning", "jsClosedThreadWarning");
                    warningElement.textContent = Language.get("wbb.thread.closedThread");
                    redactorBox.closest(".messageBody").insertBefore(warningElement, redactorBox);
                }
                // Prevent auto-open on reply.
                const checkbox = document.querySelector('#settings_text input[name="closeThread"]');
                if (checkbox) {
                    checkbox.checked = true;
                }
            }
        }
        open(threadId) {
            // Remove close icon in meta data.
            const jsClosedIcon = this.getThread(threadId).querySelector(".jsIconClosed");
            if (!jsClosedIcon) {
                throw new Error(`".jsIconClosed" icon not found for thread with id "${threadId}".`);
            }
            jsClosedIcon.closest("li").remove();
            // Remove post warning.
            const warningElement = document.querySelector(".jsClosedThreadWarning");
            if (warningElement) {
                warningElement.remove();
            }
            // Prevent auto-close on reply.
            const checkbox = document.querySelector('#settings_text input[name="closeThread"]');
            if (checkbox) {
                checkbox.checked = false;
            }
        }
        delete(threadId, _boardLink) {
            new Repeating_1.default((timer) => {
                timer.stop();
                window.location.href = _boardLink;
            }, 1000);
        }
        disable(threadId) {
            var _a;
            (_a = this.postHandler) === null || _a === void 0 ? void 0 : _a.updateAllItems({ isDisabled: 1 });
            const thread = this.getThread(threadId);
            thread.dataset.isDisabled = "1";
            thread.classList.add("messageDisabled");
        }
        enable(threadId, ignorePost = false) {
            var _a;
            if (!ignorePost) {
                (_a = this.postHandler) === null || _a === void 0 ? void 0 : _a.updateAllItems({ isDisabled: 0 });
            }
            const thread = this.getThread(threadId);
            thread.dataset.isDisabled = "0";
            thread.classList.remove("messageDisabled");
        }
        moved(_threadId) {
            window.location.reload();
        }
        showMoveNotice(_threadId) {
            window.location.reload();
        }
        sticky(threadId) {
            const contentHeaderIcon = this.getThread(threadId).querySelector(".contentHeaderIcon");
            contentHeaderIcon.appendChild(getStickyIcon());
        }
        scrape(threadId) {
            const icon = this.getThread(threadId).querySelector(".jsIconSticky");
            icon === null || icon === void 0 ? void 0 : icon.remove();
        }
        trash(threadId) {
            var _a;
            (_a = this.postHandler) === null || _a === void 0 ? void 0 : _a.updateAllItems({ isDeleted: 1 });
            this.getThread(threadId).classList.add("messageDeleted");
        }
        restore(threadId, ignorePosts = false) {
            var _a;
            if (!ignorePosts) {
                (_a = this.postHandler) === null || _a === void 0 ? void 0 : _a.updateAllItems({ isDeleted: 0 });
            }
            this.getThread(threadId).classList.remove("messageDeleted");
        }
        done(threadId) {
            const listItem = this.getThread(threadId).querySelector(".jsMarkAsDone");
            if (!listItem) {
                throw new Error(`".jsMarkAsDone" icon not found for thread with id "${threadId}".`);
            }
            const icon = listItem.querySelector(".icon");
            icon.classList.remove("fa-square-o");
            icon.classList.add("fa-check-square-o");
            const text = listItem.querySelector("span:not(.icon)");
            text.textContent = Language.get("wbb.thread.done");
        }
        undone(threadId) {
            const listItem = this.getThread(threadId).querySelector(".jsMarkAsDone");
            if (!listItem) {
                throw new Error(`".jsMarkAsDone" icon not found for thread with id "${threadId}".`);
            }
            const icon = listItem.querySelector(".icon");
            icon.classList.remove("fa-check-square-o");
            icon.classList.add("fa-square-o");
            const text = listItem.querySelector("span:not(.icon)");
            text.textContent = Language.get("wbb.thread.undone");
        }
        updateTitle(_threadId, _title) {
            // Does nothing.
        }
    }
    exports.ThreadUpdateHandler = ThreadUpdateHandler;
    function getCloseElement() {
        return Util_1.default.createFragmentFromHtml(`<li><span class="icon icon16 fa-lock jsIconClosed"></span> ${Language.get("wbb.thread.closed")}</li>`);
    }
    function getStickyIcon() {
        return Util_1.default.createFragmentFromHtml(`<span class="icon icon16 fa-thumb-tack jsTooltip jsIconSticky wbbStickyIcon" title="${Language.get("wbb.thread.sticky")}"></span>`);
    }
    exports.default = ThreadUpdateHandler;
});
