/**
 * Class and function collection for WBB
 *
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 */

/**
 * Initialize WBB namespace
 */
var WBB = { };

/**
 * Namespace for board related actions
 */
WBB.Board = { };

/**
 * Provides collapsible support for categories.
 *
 * @see	WCF.Collapsible.Remote
 */
WBB.Board.Collapsible = WCF.Collapsible.Remote.extend({
	/**
	 * @see	WCF.Collapsible.Remote._getContainers()
	 */
	_getContainers: function() {
		return $('.wbbBoardList .wbbCollapsibleCategory');
	},

	/**
	 * @see	WCF.Collapsible.Remote._getTarget()
	 */
	_getTarget: function(containerID) {
		return this._containers[containerID].children('ul').first();
	},

	/**
	 * @see	WCF.Collapsible.Remote._getButtonContainer()
	 */
	_getButtonContainer: function(containerID) {
		return this._containers[containerID].find('header h2').first();
	},

	/**
	 * @see	WCF.Collapsible.Remote._getObjectID()
	 */
	_getObjectID: function(containerID) {
		return this._containers[containerID].data('boardID');
	},

	/**
	 * @see	WCF.Collapsible.Remote._updateContent()
	 */
	_updateContent: function(containerID, newContent, newState) {
		var $newContainer = $(newContent).attr('id', containerID);

		this._containers[containerID].replaceWith($newContainer);
		this._containers[containerID] = $newContainer;
		this._initContainer(containerID, $newContainer);
	},

	/**
	 * @see	WCF.Collapsible.Remote._getAdditionalParameters()
	 */
	_getAdditionalParameters: function(containerID) {
		return {
			depth: this._containers[containerID].data('depth')
		};
	},
});

/**
 * Marks all boards as read.
 *
 * @param	object		callback
 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Board/MarkAllAsRead` instead
 */
WBB.Board.MarkAllAsRead = Class.extend({
	/**
	 * Initializes the WBB.Board.MarkAllAsRead class.
	 *
	 * @param	object		callback
	 */
	init: function(callback) {
		require(['WoltLabSuite/Forum/Ui/Board/MarkAllAsRead'], ({ BoardMarkAllAsRead }) => {
			document.querySelectorAll('.markAllAsReadButton').forEach((el) => {
				new BoardMarkAllAsRead(el, callback);
			});
		});
	}
});

/**
 * Marks a board as read.
 *
 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Board/MarkAsRead` instead
 */
WBB.Board.MarkAsRead = Class.extend({
	/**
	 * Initializes the WBB.Board.MarkAsRead class.
	 */
	init: function() {
		require(['WoltLabSuite/Forum/Ui/Board/MarkAsRead'], (BoardMarkAsRead) => BoardMarkAsRead.setup());
	}
});

/**
 * Marks all threads as read
 *
 * @param	integer		boardID
 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Board/MarkAllThreadsAsRead` instead
 */
WBB.Board.MarkAllThreadsAsRead = Class.extend({
	/**
	 * Initializes the WBB.Board.MarkAllThreadsAsRead class.
	 *
	 * @param	integer		boardID
	 */
	init: function(boardID) {
		require(['WoltLabSuite/Forum/Ui/Board/MarkAllThreadsAsRead'], (BoardMarkAllThreadsAsRead) => BoardMarkAllThreadsAsRead.setup(boardID));
	},
});

if (COMPILER_TARGET_DEFAULT) {
	/**
	 * Shows a dialog overlay to setup ignored boards.
	 *
	 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Board/Ignore` instead
	 */
	WBB.Board.IgnoreBoards = Class.extend({
		init: function () {
			require(['WoltLabSuite/Forum/Ui/Board/Ignore'], (Ignore) => Ignore.setup());
		}
	});
}
else {
	WBB.Board.IgnoreBoards = Class.extend({
		_dialog: {},
		_didInit: false,
		_proxy: {},
		init: function() {},
		_click: function() {},
		_initBoardList: function() {},
		_checkboxClick: function() {},
		_submit: function() {},
		_success: function() {}
	});
}

/**
 * Namespace for thread related actions
 */
WBB.Thread = { };

/**
 * Marks one thread as read.
 *
 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Thread/MarkAsRead` instead
 */
WBB.Thread.MarkAsRead = Class.extend({
	/**
	 * Initializes the WBB.Thread.MarkAsRead class.
	 */
	init: function() {
		require(['WoltLabSuite/Forum/Ui/Thread/MarkAsRead'], (ThreadMarkAsRead) => ThreadMarkAsRead.setup());
	}
});

if (COMPILER_TARGET_DEFAULT) {
	/**
	 * Inline editor for threads.
	 *
	 * @param        string                environment
	 * @param        integer                threadID
	 */
	WBB.Thread.Editor = {
		/**
		 * list of callbacks
		 * @var        Function[]
		 */
		_callbacks: [],

		/**
		 * editor dialog content
		 * @var        jQuery
		 */
		_dialog: null,

		/**
		 * editor dialog
		 * @var        jQuery
		 */
		_dialogContainer: null,

		/**
		 * current environment
		 * @var        string
		 */
		_environment: '',

		/**
		 * notification object
		 * @var        WCF.System.Notification
		 */
		_notification: null,

		/**
		 * action proxy
		 * @var        WCF.Action.Proxy
		 */
		_proxy: null,

		/**
		 * thread id
		 * @var        integer
		 */
		_threadID: 0,

		/**
		 * Fetches editor content on request.
		 *
		 * @param        string                environment
		 * @param        integer                threadID
		 */
		beginEdit: function (environment, threadID) {
			// unbind previous listeners to prevent them from stacking up
			this._callbacks = [];

			this._environment = environment;
			this._threadID = threadID;

			if (this._proxy === null) {
				this._proxy = new WCF.Action.Proxy({
					success: $.proxy(this._success, this)
				});
			}

			this._proxy.setOption('data', {
				actionName: 'beginEdit',
				className: 'wbb\\data\\thread\\ThreadAction',
				parameters: {
					data: {
						threadID: this._threadID
					}
				}
			});
			this._proxy.sendRequest();
		},

		/**
		 * Registers a callback.
		 *
		 * @param        object                callback
		 */
		registerCallback: function (callback) {
			this._callbacks.push(callback);
		},

		/**
		 * Handles AJAX repsonses.
		 *
		 * @param        object                data
		 * @param        string                textStatus
		 * @param        jQuery                jqXHR
		 */
		_success: function (data, textStatus, jqXHR) {
			switch (data.returnValues.actionName) {
				case 'beginEdit':
					this._showEditor(data);
					break;

				case 'saveEdit':
					this._saveEdit(data);
					break;
			}
		},

		/**
		 * Displays the editor form overlay.
		 *
		 * @param        object                data
		 */
		_showEditor: function (data) {
			if (this._dialog === null) {
				this._dialogContainer = $('<div id="threadEditor" />').data('wbbThreadEditor', this).hide().appendTo(document.body);
				this._dialog = $('<div />').appendTo(this._dialogContainer);

				// add form controls
				var $formSubmit = $('<div class="formSubmit" />').appendTo(this._dialogContainer);
				var $saveButton = $('<button class="buttonPrimary">' + WCF.Language.get('wcf.global.button.save') + '</button>').appendTo($formSubmit);
				var $cancelButton = $('<button>' + WCF.Language.get('wcf.global.button.cancel') + '</button>').appendTo($formSubmit);

				$cancelButton.click($.proxy(this._cancel, this));
				$saveButton.click($.proxy(this._save, this));
			}
			else if ($('#threadEditorLanguageIDContainer').length) {
				require(['WoltLabSuite/Core/Language/Chooser'], function (LanguageChooser) {
					LanguageChooser.removeChooser('threadLanguageID');
				});
			}

			this._dialogContainer.data('threadID', data.returnValues.threadID);
			this._dialog.html(data.returnValues.template);

			var self = this;
			this._dialog.find('#topic').keyup(function (event) {
				if (event.which === $.ui.keyCode.ENTER) {
					self._save();

					event.preventDefault();
					return false;
				}
			});

			// show dialog
			this._dialogContainer.wcfDialog({
				title: WCF.Language.get('wbb.thread.edit')
			});
			this._dialogContainer.wcfDialog('render');
		},

		/**
		 * Collects input values for save.
		 */
		_save: function () {
			var $values = {};

			// get all fieldsets
			var $fieldsets = this._dialog.find('fieldset, .section');
			if (!$fieldsets.length) {
				return;
			}

			// search for fieldsets
			$fieldsets.each($.proxy(function (index, fieldset) {
				var $fieldset = $(fieldset);
				var $identifier = $fieldset.attr('id').replace(/^threadEditor/, '').toLowerCase();

				$values[$identifier] = this._getValues($fieldset);
			}, this));

			// execute callbacks
			for (var $i = 0, $length = this._callbacks.length; $i < $length; $i++) {
				$values = this._callbacks[$i].getValues(this._dialog, $values);
			}

			if ($.getLength($values)) {
				this._proxy.setOption('data', {
					actionName: 'saveEdit',
					className: 'wbb\\data\\thread\\ThreadAction',
					parameters: {
						data: {
							values: $values,
							threadID: this._threadID
						}
					}
				});
				this._proxy.sendRequest();
			}
		},

		/**
		 * Show success notification and optionally reload current page.
		 *
		 * @param        object                data
		 */
		_saveEdit: function (data) {
			if (data.returnValues.errors && $.getLength(data.returnValues.errors)) {
				this._dialog.find('small.innerError').remove();

				// execute callbacks
				for (var $i = 0, $length = this._callbacks.length; $i < $length; $i++) {
					if (typeof this._callbacks[$i].showErrors === 'function') {
						this._callbacks[$i].showErrors(data.returnValues.errors);
					}
				}

				return;
			}

			if (this._notification === null) {
				this._notification = new WCF.System.Notification(WCF.Language.get('wcf.global.success.edit'));
			}

			if (data.returnValues.forceReload) {
				this._notification.show(function () {
					window.location.reload();
				}, undefined, WCF.Language.get('wcf.global.success.edit'));
			}
			else {
				// perform inline editing
				for (var $i in this._callbacks) {
					this._callbacks[$i].saved(this._environment, this._threadID, data);
				}

				this._dialogContainer.wcfDialog('close');
				this._notification.show();
			}
		},

		/**
		 * Returns a list of all input values within a fieldset.
		 *
		 * @param        jQuery                fieldset
		 * @return        object
		 */
		_getValues: function (fieldset) {
			var $values = {};

			fieldset.find('input:not([type=radio])').each(function (index, input) {
				var $input = $(input);

				if ($input.attr('type') === 'checkbox') {
					$values[$input.attr('id')] = ($input.prop('checked') ? 1 : 0);
				}
				else {
					$values[$input.attr('id')] = $input.val();
				}
			});

			fieldset.find('input[type=radio]:checked').each(function (index, input) {
				var $input = $(input);
				$values[$input.attr('name')] = $input.val();
			});

			fieldset.find('select').each(function (index, select) {
				var $select = $(select);
				$values[$select.attr('id')] = $select.val();
			});

			return $values;
		},

		/**
		 * Closes the editor.
		 */
		_cancel: function () {
			this._dialogContainer.wcfDialog('close');
		}
	};

	/**
	 * Default thread editor handler.
	 */
	WBB.Thread.Editor.Default = Class.extend({
		/**
		 * Initializes the WBB.Thread.Editor.Default class.
		 */
		init: function () {
			$('#threadEditor').data('wbbThreadEditor').registerCallback(this);

			this._handleThreadType();
		},

		/**
		 * Handles thread type.
		 */
		_handleThreadType: function () {
			var $label = $('#threadEditorAnnouncementLabel');
			var $container = $('#threadEditorAnnouncementContainer');

			var $update = function (type, init) {
				switch (type) {
					/* default */
					case 0:
					/* sticky */
					case 1:
						$label.hide();
						$container.hide();

						if (!init) $('#threadEditor').wcfDialog('render');
						break;

					/* announcement */
					case 2:
						$label.show();
						$container.show();

						if (!init) $('#threadEditor').wcfDialog('render');
						break;
				}
			};

			var $inputElements = $('#threadEditor input[name=type]').click(function (event) {
				$update(parseInt($(event.currentTarget).val()), false);
			});

			// set type on init
			$inputElements.each(function (index, element) {
				var $element = $(element);
				if ($element.is(':checked')) {
					$update(parseInt($element.val()), true);

					return false;
				}
			});
		},

		/**
		 * Handles click event.
		 *
		 * @param        object                event
		 */
		_click: function (event) {
			this._type = parseInt($(event.currentTarget).prop('value'));

			this._update();
		},

		/**
		 * Updates board ids container state.
		 */
		_update: function () {
			switch (this._type) {

			}
		},

		/**
		 * Does nothing.
		 */
		getValues: function (dialog, values) {
			return values;
		},

		/**
		 * Handles successful thread edits.
		 *
		 * @param        string                environment
		 * @param        integer                threadID
		 * @param        object                data
		 */
		saved: function (environment, threadID, data) {
			if (data.returnValues['default'] === undefined) {
				return;
			}

			if (environment === 'board') {
				$('#thread' + threadID + ' .columnSubject > h3 > a').text(data.returnValues['default'].topic);
			}
			else {
				$('.contentHeader[data-thread-id="' + threadID + '"] .contentTitle').text(data.returnValues['default'].topic);
			}
		},

		/**
		 * Displays error messages.
		 *
		 * @param        object                errors
		 */
		showErrors: function (errors) {
			if (errors['default']) {
				for ($fieldName in errors['default']) {
					$('<small class="innerError">' + errors['default'][$fieldName] + '</small>').insertAfter($('#threadEditorDefault #' + $fieldName));
				}
			}
		}
	});

	/**
	 * Label chooser for dialog thread editor.
	 *
	 * @see        WCF.Label.Chooser
	 */
	WBB.Thread.Editor.LabelChooser = WCF.Label.Chooser.extend({
		/**
		 * @see        WCF.Label.Chooser.init()
		 */
		init: function (selectedLabelIDs, containerSelector, submitButtonSelector, showWithoutSelection) {
			this._super(selectedLabelIDs, containerSelector, submitButtonSelector, showWithoutSelection);

			$('#threadEditor').data('wbbThreadEditor').registerCallback(this);
		},

		/**
		 * Adds label ids to dialog data.
		 *
		 * @param        jQuery                dialog
		 * @param        object                values
		 * @return        values
		 */
		getValues: function (dialog, values) {
			values.label = {};

			for (var $groupID in this._groups) {
				var $group = this._groups[$groupID];
				if ($group.data('labelID') !== undefined) {
					values.label[$groupID] = $group.data('labelID');
				}
			}

			return values;
		},

		/**
		 * Handles successful thread edits.
		 *
		 * @param        string                environment
		 * @param        integer                threadID
		 * @param        object                data
		 */
		saved: function (environment, threadID, data) {
			if (data.returnValues.label === undefined) {
				return;
			}

			var $i, $label, $listItem;
			var $labels = data.returnValues.label.labels;
			var $labelList = null;
			if (environment === 'board') {
				var $column = $('#thread' + threadID + ' > .columnSubject');
				$labelList = $column.children('.labelList');
				if ($labelList.length) {
					if ($labels.length) {
						// remove existing labels
						$labelList.empty();
					}
					else {
						// remove label list
						$labelList.remove();
					}
				}
				else if ($labels.length) {
					// create label list
					$labelList = $('<ul class="labelList" />').prependTo($column);
				}

				for ($i = 0; $i < $labels.length; $i++) {
					$label = $labels[$i];
					$listItem = $('<li><a href="' + $label.link + '" class="badge label ' + $label.cssClassName + '">' + WCF.String.escapeHTML($label.label) + '</a></li>').appendTo($labelList);
					$listItem.before(' ');
				}
			}
			else {
				var $header = $('header.wbbThread[data-thread-id=' + threadID + ']');
				$labelList = $header.find('.labelList');
				if ($labelList.length) {
					if ($labels.length) {
						// remove existing labels
						$labelList.empty();
					}
					else {
						// remove label list
						$labelList.parent().remove();
					}
				}
				else if ($labels.length) {
					// create label list
					$listItem = $('<li><span class="icon icon16 fa-tags"></span> </li>').prependTo($header.find('.contentHeaderMetaData'));
					$labelList = $('<ul class="labelList" />').appendTo($listItem);
				}

				for ($i = 0; $i < $labels.length; $i++) {
					$label = $labels[$i];
					$listItem = $('<li><span class="badge label ' + $label.cssClassName + '">' + WCF.String.escapeHTML($label.label) + '</span></li>').appendTo($labelList);
					$listItem.after(' ');
				}
			}
		},

		/**
		 * Displays error messages.
		 *
		 * @param        object                errors
		 */
		showErrors: function (errors) {
			if (errors.label) {
				var $container = $('#threadEditorLabel');
				$container.find('.labelList').each(function (index, labelList) {
					var $labelList = $(labelList);
					var $groupID = $labelList.data('objectID');

					if (errors.label.labelIDs[$groupID]) {
						$('<small class="innerError">' + WCF.Language.get('wcf.label.error.missing') + '</small>').insertAfter($labelList);
					}
				});
			}
		}
	});

	/**
	 * Language chooser for dialog thread editor.
	 *
	 * @see        WCF.Language.Chooser
	 */
	WBB.Thread.Editor.Language = WCF.Language.Chooser.extend({
		/**
		 * @see        WCF.Language.Chooser.init()
		 */
		init: function (containerID, inputFieldID, languageID, languages, callback, allowEmptyValue) {
			this._languages = languages;

			this._super(containerID, inputFieldID, languageID, languages, callback, allowEmptyValue);

			$('#threadEditor').data('wbbThreadEditor').registerCallback(this);
		},

		/**
		 * Adds label ids to dialog data.
		 *
		 * @param        jQuery                dialog
		 * @param        object                values
		 * @return        values
		 */
		getValues: function (dialog, values) {
			if (values.default.threadLanguageID) {
				values.default.languageID = values.default.threadLanguageID;
				delete values.default.threadLanguageID;
			}

			return values;
		},

		/**
		 * Handles successful thread edits.
		 *
		 * @param        string                environment
		 * @param        integer                threadID
		 * @param        object                data
		 */
		saved: function (environment, threadID, data) {
			if (environment === 'board') {
				var iconFlag = $('#thread' + threadID + ' > .columnSubject .statusDisplay .iconFlag');

				iconFlag.attr('src', this._languages[data.returnValues.default.languageID].iconPath);
			}
		}
	});

	/**
	 * Tag list for dialog thread editor.
	 *
	 * @see        WCF.Tagging.TagList
	 */
	WBB.Thread.Editor.TagList = Class.extend({
		_api: null,

		init: function () {
			$('#threadEditor').data('wbbThreadEditor').registerCallback(this);

			require(['WoltLabSuite/Core/Ui/ItemList', 'Ui/SimpleDropdown'], (function (UiItemList, UiSimpleDropdown) {
				// destroy previous tag search dropdowns
				UiSimpleDropdown.destroy('tagSearchInputThread');

				this._api = UiItemList;
			}).bind(this));
		},

		/**
		 * Adds tags to dialog data.
		 *
		 * @param        jQuery                dialog
		 * @param        object                values
		 * @return        values
		 */
		getValues: function (dialog, values) {
			delete values.tags;

			var data = [];
			var tmp = this._api.getValues('tagSearchInputThread');
			for (var i = 0, length = tmp.length; i < length; i++) {
				data.push(tmp[i].value);
			}

			values.tag = (data.length ? data : ['__wcf_noTags']);

			return values;
		},

		/**
		 * Handles successful thread edits.
		 *
		 * @param        string                environment
		 * @param        integer                threadID
		 * @param        object                data
		 */
		saved: function (environment, threadID, data) {
			/* does nothing */
		}
	});

	/**
	 * Inline editor for threads.
	 *
	 * @param        string                elementSelector
	 */
	WBB.Thread.InlineEditor = WCF.InlineEditor.extend({
		/**
		 * is `true` if advanced mode is disabled
		 * @var	boolean
		 */
		_advancedIsDisabled: false,

		/**
		 * board id
		 * @var        integer
		 */
		_boardID: 0,

		/**
		 * current editor environment
		 * @var        string
		 */
		_environment: 'thread',

		/**
		 * list of permissions
		 * @var        object
		 */
		_permissions: {},

		/**
		 * redirect URL
		 * @var        string
		 */
		_redirectURL: '',

		/**
		 * thread update handler
		 * @var        WBB.Thread.UpdateHandler
		 */
		_updateHandler: null,

		/**
		 * @see        WCF.InlineEditor._setOptions()
		 */
		_setOptions: function () {
			this._boardID = 0;
			this._environment = 'thread';

			this._options = [
				// isDone
				{label: WCF.Language.get('wbb.thread.edit.done'), optionName: 'done'},
				{label: WCF.Language.get('wbb.thread.edit.undone'), optionName: 'undone'},

				// isClosed
				{label: WCF.Language.get('wbb.thread.edit.close'), optionName: 'close'},
				{label: WCF.Language.get('wbb.thread.edit.open'), optionName: 'open'},

				// isSticky
				{label: WCF.Language.get('wbb.thread.edit.sticky'), optionName: 'sticky'},
				{label: WCF.Language.get('wbb.thread.edit.scrape'), optionName: 'scrape'},

				// isDisabled
				{label: WCF.Language.get('wbb.thread.edit.enable'), optionName: 'enable'},
				{label: WCF.Language.get('wbb.thread.edit.disable'), optionName: 'disable'},

				// move thread
				{label: WCF.Language.get('wbb.thread.edit.move'), optionName: 'move'},
				{label: WCF.Language.get('wbb.thread.edit.removeLink'), optionName: 'removeLink'},

				// isDeleted
				{label: WCF.Language.get('wbb.thread.edit.trash'), optionName: 'trash'},
				{label: WCF.Language.get('wbb.thread.edit.restore'), optionName: 'restore'},
				{label: WCF.Language.get('wbb.thread.edit.delete'), optionName: 'delete'},

				// divider
				{optionName: 'divider'},

				// Marking the thread.
				{label: WCF.Language.get('wbb.thread.edit.markThread'), optionName: 'markThread'},
				{label: WCF.Language.get('wbb.thread.edit.unmarkThread'), optionName: 'unmarkThread'},

				// Marking all posts on the current page.
				{label: WCF.Language.get('wbb.thread.edit.markPosts'), optionName: 'markPosts'},

				// overlay
				{
					label: WCF.Language.get('wbb.thread.edit.advanced'),
					optionName: 'advanced',
					isQuickOption: true
				}
			];
		},

		/**
		 * Returns current update handler.
		 *
		 * @return        WBB.Thread.UpdateHandler
		 */
		setUpdateHandler: function (updateHandler) {
			this._updateHandler = updateHandler;
		},

		/**
		 * @see        WCF.InlineEditor._getTriggerElement()
		 */
		_getTriggerElement: function (element) {
			return element.find('.jsThreadInlineEditor');
		},

		/**
		 * @see        WCF.InlineEditor._validate()
		 */
		_validate: function (elementID, optionName) {
			var $threadID = $('#' + elementID).data('threadID');

			// links for moved threads only support their removal
			if (this._updateHandler.getValue($threadID, 'isLink')) {
				if (optionName === 'removeLink' && this._getPermission('canMoveThread')) {
					return true;
				}

				return false;
			}

			switch (optionName) {
				// isOpen
				case 'close':
				case 'open':
					if (!this._getPermission('canCloseThread')) {
						return false;
					}

					if (optionName === 'open') {
						return (this._updateHandler.getValue($threadID, 'isClosed'));
					}
					else {
						return !(this._updateHandler.getValue($threadID, 'isClosed'));
					}
					break;

				//isDeleted
				case 'delete':
					if (!this._getPermission('canDeleteThreadCompletely')) {
						return false;
					}

					return (this._updateHandler.getValue($threadID, 'isDeleted'));
					break;

				case 'done':
				case 'undone':
					if (!this._updateHandler.getValue($threadID, 'canMarkAsDone')) {
						return false;
					}

					if (optionName === 'undone') {
						return (this._updateHandler.getValue($threadID, 'isDone'));
					}
					else {
						return !(this._updateHandler.getValue($threadID, 'isDone'));
					}
					break;

				case 'restore':
					if (!this._getPermission('canRestoreThread')) {
						return false;
					}

					return (this._updateHandler.getValue($threadID, 'isDeleted'));
					break;

				case 'trash':
					if (!this._getPermission('canDeleteThread')) {
						return false;
					}

					return !(this._updateHandler.getValue($threadID, 'isDeleted'));
					break;

				// isSticky
				case 'sticky':
				case 'scrape':
					if (!this._getPermission('canPinThread') || (this._updateHandler.getValue($threadID, 'isAnnouncement'))) {
						return false;
					}

					if (optionName === 'scrape') {
						return (this._updateHandler.getValue($threadID, 'isSticky'));
					}
					else {
						return !(this._updateHandler.getValue($threadID, 'isSticky'));
					}
					break;

				// isDisabled
				case 'enable':
					if (!this._getPermission('canEnableThread')) {
						return false;
					}

					if (this._updateHandler.getValue($threadID, 'isDeleted')) {
						return false;
					}

					return (this._updateHandler.getValue($threadID, 'isDisabled'));
					break;
				case 'disable':
					if (!this._getPermission('canEnableThread')) {
						return false;
					}

					if (this._updateHandler.getValue($threadID, 'isDeleted')) {
						return false;
					}

					return !(this._updateHandler.getValue($threadID, 'isDisabled'));
					break;

				// move
				case 'move':
					return this._getPermission('canMoveThread');
					break;

				case 'markThread':
					if (this._environment !== "thread") {
						return false;
					}

					var clipboardMark = document.getElementById('wbbThreadClipboardMark');

					return clipboardMark && !clipboardMark.checked;
					break;

				case 'unmarkThread':
					if (this._environment !== "thread") {
						return false;
					}

					var clipboardMark = document.getElementById('wbbThreadClipboardMark');

					return clipboardMark && clipboardMark.checked;
					break;

				case 'markPosts':
					if (this._environment !== "thread") {
						return false;
					}

					// Only offer the action if there is at least one post that its not marked.
					return Array.from(
						document.querySelectorAll('.wbbThreadPostList .wbbPost .jsClipboardItem')
					).some(
						(input) => !input.checked
					);
					break;

				case 'advanced':
					return !this._advancedIsDisabled;
					break;
			}

			return false;
		},

		/**
		 * @see        WCF.InlineEditor._execute()
		 */
		_execute: function (elementID, optionName) {
			// abort if option is invalid or not accessible
			if (!this._validate(elementID, optionName)) {
				return false;
			}

			switch (optionName) {
				case 'close':
				case 'open':
					var $isClosed = (optionName === 'open') ? 0 : 1;
					this._updateThread(elementID, optionName, {isClosed: $isClosed});
					break;

				case 'sticky':
				case 'scrape':
					var $isSticky = (optionName === 'scrape') ? 0 : 1;
					this._updateThread(elementID, optionName, {isSticky: $isSticky});
					break;

				case 'done':
				case 'undone':
					var $isDone = (optionName === 'done') ? 1 : 0;
					this._updateThread(elementID, optionName, {isDone: $isDone});
					break;

				case 'enable':
					require(['WoltLabSuite/Forum/Ui/Thread/Enable'], (function(UiThreadEnable) {
						UiThreadEnable.init(
							this._elements[elementID].data('threadID'),
							(function(isEnabled, updateTime) {
								if (isEnabled) this._updateThread(elementID, optionName, {isDisabled: 0, updateTime: updateTime});
								else window.location.reload();
							}).bind(this)
						);
					}).bind(this));
					break;

				case 'disable':
					this._updateThread(elementID, optionName, {isDisabled: 1});
					break;

				case 'move':
					var $threadID = this._elements[elementID].data('threadID');

					WBB.Thread.MoveHandler.prepare([$threadID], $.proxy(function (data) {
						this._updateHandler.update($threadID, data.returnValues.threadData[$threadID]);
					}, this), this._boardID, this._environment);
					break;

				case 'removeLink':
					this._updateThread(elementID, optionName, {removeLink: 1});
					break;

				case 'delete':
					var self = this;
					WCF.System.Confirmation.show(WCF.Language.get('wbb.thread.confirmDelete'), function (action) {
						if (action === 'confirm') {
							self._updateThread(elementID, optionName, {deleted: 1});
						}
					});
					break;

				case 'restore':
					this._updateThread(elementID, optionName, {isDeleted: 0});
					break;

				case 'trash':
					var self = this;
					WCF.System.Confirmation.show(WCF.Language.get('wbb.thread.confirmTrash'), function (action) {
						if (action === 'confirm') {
							self._updateThread(elementID, optionName, {
								isDeleted: 1,
								reason: $('#wcfSystemConfirmationContent').find('textarea').val()
							});
						}
					}, {}, $('<div class="section"><dl><dt>' + WCF.Language.get('wbb.thread.confirmTrash.reason') + '</dt><dd><textarea cols="40" rows="4" /></dd></dl></div>'));
					break;

				case 'markThread':
				case 'unmarkThread':
					document.getElementById('wbbThreadClipboardMark').click();

					break;

				case 'markPosts':
					document.querySelectorAll('.wbbThreadPostList .wbbPost .jsClipboardItem').forEach((input) => {
						if (!input.checked) {
							input.click();
						}
					});
					break;

				case 'advanced':
					WBB.Thread.Editor.beginEdit(this._environment, this._elements[elementID].data('threadID'));
					break;

				default:
					return false;
					break;
			}

			return true;
		},

		/**
		 * Updates thread properties.
		 *
		 * @param        string                elementID
		 * @param        string                optionName
		 * @param        object                data
		 */
		_updateThread: function (elementID, optionName, data) {
			if (optionName === 'delete' || optionName === 'removeLink') {
				var self = this;
				var $threadID = this._elements[elementID].data('threadID');

				new WCF.Action.Proxy({
					autoSend: true,
					data: {
						actionName: optionName,
						className: 'wbb\\data\\thread\\ThreadAction',
						objectIDs: [$threadID]
					},
					success: function (data) {
						self._updateHandler.update($threadID, data.returnValues.threadData[$threadID]);
					}
				});
			}
			else {
				this._updateData.push({
					data: data,
					elementID: elementID,
					optionName: optionName
				});

				this._proxy.setOption('data', {
					actionName: optionName,
					className: 'wbb\\data\\thread\\ThreadAction',
					objectIDs: [this._elements[elementID].data('threadID')],
					parameters: {
						data: data
					}
				});
				this._proxy.sendRequest();
			}
		},

		/**
		 * @see        WCF.InlineEditor._updateState()
		 */
		_updateState: function () {
			this._notification.show();

			for (var $i = 0, $length = this._updateData.length; $i < $length; $i++) {
				var $data = this._updateData[$i];

				if (($data.data.isSticky !== undefined) && this._redirectURL) {
					window.location = this._redirectURL;
					return;
				}
				else if ($data.data.isDisabled !== undefined && $data.data.isDisabled === 0) {
					window.location.reload();
					return;
				}

				var $threadID = $('#' + $data.elementID).data('threadID');
				this._updateHandler.update($threadID, $data.data);
			}
		},

		/**
		 * Returns a specific permission.
		 *
		 * @param        string                permission
		 * @return        integer
		 */
		_getPermission: function (permission) {
			if (this._permissions[permission]) {
				return this._permissions[permission];
			}

			return 0;
		},

		/**
		 * Disables (or) enables the advanced edit options.
		 *
		 * @param	boolean		disable
		 */
		disableAdvancedOptions: function(disable) {
			if (disable === undefined) disable = true;

			this._advancedIsDisabled = !!disable;
		},

		/**
		 * Sets current environment.
		 *
		 * @param        string                environment
		 * @param        integer                boardID
		 * @param        string                redirectURL
		 */
		setEnvironment: function (environment, boardID, redirectURL) {
			if (environment !== 'board') {
				environment = 'thread';
			}

			this._boardID = (boardID) ? boardID : 0;
			this._environment = environment;
			this._redirectURL = redirectURL;
		},

		/**
		 * Sets a permission.
		 *
		 * @param        string                permission
		 * @param        integer                value
		 */
		setPermission: function (permission, value) {
			this._permissions[permission] = value;
		},

		/**
		 * Sets permissions.
		 *
		 * @param        object                permissions
		 */
		setPermissions: function (permissions) {
			for (var $permission in permissions) {
				this.setPermission($permission, permissions[$permission]);
			}
		}
	});

	/**
	 * Provides extended actions for thread clipboard actions.
	 */
	WBB.Thread.Clipboard = Class.extend({
		/**
		 * board id
		 * @var        integer
		 */
		_boardID: 0,

		/**
		 * current environment
		 * @var        string
		 */
		_environment: 'board',

		/**
		 * thread update handler
		 * @var        WBB.Thread.UpdateHandler
		 */
		_updateHandler: null,

		/**
		 * Initializes a new WBB.Thread.Clipboard object.
		 *
		 * @param        WBB.Thread.UpdateHandler        updateHandler
		 * @param        string                                environment
		 * @param        integer                                boardID
		 */
		init: function (updateHandler, environment, boardID) {
			this._updateHandler = updateHandler;
			this._environment = environment;
			this._boardID = (boardID) ? boardID : 0;

			require(['EventHandler'], function (EventHandler) {
				EventHandler.add('com.woltlab.wcf.clipboard', 'com.woltlab.wbb.thread', this._clipboardAction.bind(this));
			}.bind(this));
		},

		/**
		 * Reacts to executed clipboard actions.
		 *
		 * @param        {Object}        actionData        data of the executed clipboard action
		 */
		_clipboardAction: function (actionData) {
			switch (actionData.data.actionName) {
				case 'com.woltlab.wbb.thread.assignLabel':
					WBB.Thread.AssignLabelHandler.prepare(actionData.data.parameters);
					break;

				case 'com.woltlab.wbb.thread.copy':
					WBB.Thread.CopyHandler.prepare(actionData.data.parameters);
					break;

				case 'com.woltlab.wbb.thread.merge':
					var $mergeHandler = new WBB.Thread.MergeHandler(actionData.data.parameters.objectIDs);
					$mergeHandler.load();
					break;

				case 'com.woltlab.wbb.thread.move':
					WBB.Thread.MoveHandler.prepare(actionData.data.parameters.objectIDs, $.proxy(this._move, this), this._boardID, this._environment);
					break;
			}

			if (actionData.responseData !== null) {
				this._evaluateResponse(undefined, actionData.responseData, 'com.woltlab.wbb.thread', actionData.data.actionName, null);
			}
		},

		/**
		 * Handles 'move thread' action.
		 *
		 * @param        object                data
		 */
		_move: function (data) {
			window.location.reload();
		},

		/**
		 * Evaluates AJAX responses.
		 *
		 * @param        object                event
		 * @param        object                data
		 * @param        string                type
		 * @param        string                actionName
		 * @param        object                parameters
		 */
		_evaluateResponse: function (event, data, type, actionName, parameters) {
			// ignore unrelated events
			if (type !== 'com.woltlab.wbb.thread') {
				return;
			}

			if (!data.returnValues.threadData || !$.getLength(data.returnValues.threadData)) {
				return;
			}

			// loop through threads
			for (var $threadID in data.returnValues.threadData) {
				this._updateHandler.update($threadID, data.returnValues.threadData[$threadID], false);
			}

			WCF.Clipboard.reload();
		}
	});

	/**
	 * @deprecated	5.5, use `WoltLabSuite/Forum/Handler/Thread/UpdateHandler` instead
	 */
	WBB.Thread.UpdateHandler = Class.extend({
		_updateHandlerPromise: null,
		_updateHandler: null,

		init: function () {
			console.warn("WBB.Thread.UpdateHandler is deprecated since version 5.5. Use WoltLabSuite/Forum/Handler/Thread/UpdateHandler instead.");
		},

		setMarkAsDoneHandler: function (markAsDoneHandler) {
			// Does nothing.
		},

		update: function (threadID, data, reloadClipboard) {
			this._updateHandler.update(threadID, data, reloadClipboard);
		},

		getValue: function (threadID, property) {
			return this._updateHandler.getValue(threadID, property);
		}
	});

	/**
	 * @deprecated	5.5, use `WoltLabSuite/Forum/Handler/Thread/BoardUpdateHandler` instead
	 */
	WBB.Thread.UpdateHandler.Board = WBB.Thread.UpdateHandler.extend({
		init: function() {
			console.warn("WBB.Thread.UpdateHandler.Board is deprecated since version 5.5. Use WoltLabSuite/Forum/Handler/Thread/BoardUpdateHandler instead.");

			this._updateHandlerPromise = require(['WoltLabSuite/Forum/Handler/Thread/BoardUpdateHandler']).then(([{BoardUpdateHandler}]) => {
				this._updateHandler = new BoardUpdateHandler();

				return this._updateHandler;
			});
		}
	});

	/**
	 * @deprecated	5.5, use `WoltLabSuite/Forum/Handler/Thread/ThreadUpdateHandler` instead
	 */
	WBB.Thread.UpdateHandler.Thread = WBB.Thread.UpdateHandler.extend({
		init: function(boardID) {
			console.warn("WBB.Thread.UpdateHandler.Thread is deprecated since version 5.5. Use WoltLabSuite/Forum/Handler/Thread/ThreadUpdateHandler instead.");

			this._updateHandlerPromise = require(['WoltLabSuite/Forum/Handler/Thread/ThreadUpdateHandler']).then(([{ThreadUpdateHandler}]) => {
				this._updateHandler = new ThreadUpdateHandler(boardID);

				return this._updateHandler;
			});
		},

		setPostHandler: function (postManager) {
			this._updateHandlerPromise.then((uh) => uh.setPostHandler(postManager));
		},

		update: function (threadID, data) {
			this._updateHandler.update(threadID, data);
		}
	});

	/**
	 * Merges multiple threads.
	 *
	 * @param        array<integer>                objectIDs
	 */
	WBB.Thread.MergeHandler = Class.extend({
		/**
		 * action class name
		 * @var        string
		 */
		_className: '',

		/**
		 * dialog overlay
		 * @var        jQuery
		 */
		_dialog: null,

		/**
		 * dialog title
		 * @var        string
		 */
		_dialogTitle: '',

		/**
		 * list of object ids
		 * @var        array<integer>
		 */
		_objectIDs: [],

		/**
		 * action proxy
		 * @var        WCF.Action.Proxy
		 */
		_proxy: null,

		/**
		 * success message after merging
		 * @var        string
		 */
		_successMessage: '',

		/**
		 * Initializes the WBB.Thread.MergeHandler object.
		 *
		 * @param        array<integer>                objectIDs
		 */
		init: function (objectIDs) {
			this._className = 'wbb\\data\\thread\\ThreadAction';
			this._dialogTitle = WCF.Language.get('wbb.thread.edit.merge');
			this._objectIDs = objectIDs;
			this._successMessage = WCF.Language.get('wbb.thread.edit.merge.success');

			this._proxy = new WCF.Action.Proxy({
				success: $.proxy(this._success, this)
			});
		},

		/**
		 * Loads the merge dialog.
		 */
		load: function () {
			this._proxy.setOption('data', {
				actionName: 'prepareMerge',
				className: this._className,
				objectIDs: this._objectIDs
			});
			this._proxy.sendRequest();
		},

		/**
		 * Handles successful AJAX requests.
		 *
		 * @param        object                data
		 * @param        string                textStatus
		 * @param        jQuery                jqXHR
		 */
		_success: function (data, textStatus, jqXHR) {
			if (data.returnValues.redirectURL) {
				var $notification = new WCF.System.Notification(this._successMessage);
				$notification.show(function () {
					window.location = data.returnValues.redirectURL;
				});

				return;
			}

			if (this._dialog === null) {
				this._dialog = $('<div />').hide().appendTo(document.body);
				this._dialog.html(data.returnValues.template);
				this._dialog.wcfDialog({
					title: this._dialogTitle
				});
			}
			else {
				this._dialog.html(data.returnValues.template);
				this._dialog.wcfDialog('show');
			}

			var $button = this._dialog.find('.formSubmit > button[data-type=submit]').disable().click($.proxy(this._submit, this));
			this._dialog.find('input[type=radio]').change(function () {
				$button.enable();
			});
		},

		/**
		 * Submits the merge dialog.
		 */
		_submit: function () {
			this._dialog.find('.formSubmit > button[data-type=submit]').disable();

			this._proxy.setOption('data', {
				actionName: 'merge',
				className: this._className,
				objectIDs: this._objectIDs,
				parameters: this._getParameters()
			});
			this._proxy.sendRequest();
		},

		/**
		 * Returns parameters for merging.
		 *
		 * @return        object
		 */
		_getParameters: function () {
			return {
				threadID: this._dialog.find('input[type=radio]:checked').val()
			};
		}
	});

	/**
	 * Assigns labels to threads.
	 */
	WBB.Thread.AssignLabelHandler = {
		/**
		 * board id
		 * @var        integer
		 */
		_boardID: 0,

		/**
		 * dialog overlay
		 * @var        jQuery
		 */
		_dialog: null,

		/**
		 * list of thread ids
		 * @var        array<integer>
		 */
		_objectIDs: [],

		/**
		 * Shows the assignment form.
		 *
		 * @param        object                parameters
		 */
		prepare: function (parameters) {
			this._boardID = parameters.boardID;
			this._objectIDs = parameters.objectIDs;

			if (this._dialog === null) {
				this._dialog = $('<div />').appendTo(document.body);
				this._dialog.html(parameters.template);
				this._dialog.wcfDialog({
					title: WCF.Language.get('wbb.thread.edit.assignLabel')
				});
			}
			else {
				this._dialog.html(parameters.template);
				this._dialog.wcfDialog('open');
			}

			this._dialog.find('.formSubmit > .buttonPrimary').click($.proxy(this._click, this));
		},

		/**
		 * Handles clicks on the submit button.
		 */
		_click: function () {
			var $labelIDs = {};
			this._dialog.find('.labelList > .dropdown').each(function (index, dropdown) {
				var $dropdown = $(dropdown);
				if ($dropdown.data('labelID')) {
					$labelIDs[$dropdown.data('groupID')] = $dropdown.data('labelID');
				}
			});

			new WCF.Action.Proxy({
				autoSend: true,
				data: {
					actionName: 'assignLabel',
					className: 'wbb\\data\\thread\\ThreadAction',
					objectIDs: this._objectIDs,
					parameters: {
						boardID: this._boardID,
						labelIDs: $labelIDs
					}
				},
				success: $.proxy(this._success, this)
			});
		},

		/**
		 * Handles successful AJAX requests.
		 *
		 * @param        object                data
		 * @param        string                textStatus
		 * @param        jQuery                jqXHR
		 */
		_success: function (data, textStatus, jqXHR) {
			var $labels = data.returnValues.labels;

			for (var $i = 0; $i < data.objectIDs.length; $i++) {
				var $column = $('#thread' + data.objectIDs[$i] + ' > .columnSubject');
				var $labelList = $column.children('.labelList');
				if ($labelList.length) {
					if ($labels.length) {
						// remove existing labels
						$labelList.empty();
					}
					else {
						// remove label list
						$labelList.remove();
					}
				}
				else if ($labels.length) {
					// create label list
					$labelList = $('<ul class="labelList" />').prependTo($column);
				}

				for (var $j = 0; $j < $labels.length; $j++) {
					var $label = $labels[$j];
					var $listItem = $('<li><a href="' + $label.link + '" class="badge label ' + $label.cssClassName + '">' + WCF.String.escapeHTML($label.label) + '</a></li>').appendTo($labelList);
					$listItem.before(' ');
				}
			}

			this._dialog.wcfDialog('close');
			WCF.Clipboard.reload();

			new WCF.System.Notification().show();
		}
	};

	/**
	 * Copies selected threads.
	 */
	WBB.Thread.CopyHandler = {
		/**
		 * target board id
		 * @var        integer
		 */
		_boardID: 0,

		/**
		 * current thread id index
		 * @var        integer
		 */
		_i: 0,

		/**
		 * list of thread ids
		 * @var        array<integer>
		 */
		_objectIDs: [],

		/**
		 * Prepares copying of posts.
		 *
		 * @param        object                parameters
		 */
		prepare: function (parameters) {
			this._boardID = parameters.boardID;
			this._i = 0;
			this._objectIDs = parameters.objectIDs;

			this.copy();
		},

		/**
		 * Copies a thread.
		 *
		 * @param        boolean                next
		 */
		copy: function (next) {
			if (next === true) {
				this._i++;
			}

			var $title = WCF.Language.get('wbb.thread.copy.title', {
				count: this._objectIDs.length,
				item: this._i + 1
			});
			if (this._i + 1 == this._objectIDs.length) {
				// last iteration
				new WCF.System.Worker('copy', 'wbb\\data\\thread\\ThreadAction', $title, {
					boardID: this._boardID,
					sourceThreadID: this._objectIDs[this._i]
				});
			}
			else {
				// there are more threads to copy, simply restart with the next thread
				new WCF.System.Worker('copy', 'wbb\\data\\thread\\ThreadAction', $title, {
					boardID: this._boardID,
					sourceThreadID: this._objectIDs[this._i]
				}, function (worker) {
					// force close and skip all this fancy stuff
					worker._dialog.wcfDialog('option', 'onClose', null);
					worker._dialog.wcfDialog('option', 'closeConfirmMessage', null);

					worker._dialog.wcfDialog('close');

					WBB.Thread.CopyHandler.copy(true);
				});
			}
		}
	};

	/**
	 * Generic implementation to move threads.
	 */
	WBB.Thread.MoveHandler = {
		/**
		 * board id
		 * @var        integer
		 */
		_boardID: 0,

		/**
		 * callback object
		 * @var        object
		 */
		_callback: null,

		/**
		 * dialog overlay
		 * @var        jQuery
		 */
		_dialog: null,

		/**
		 * initialization state
		 * @var        boolean
		 */
		_didInit: false,

		/**
		 * current editor environment
		 * @var        string
		 */
		_environment: '',

		/**
		 * action proxy
		 * @var        WCF.Action.Proxy
		 */
		_proxy: null,

		/**
		 * list of thread ids
		 * @var        array<integer>
		 */
		_threadIDs: [],

		/**
		 * Initializes WBB.Thread.MoveHandler on first use.
		 */
		_init: function () {
			this._dialog = $('<div />').hide().appendTo(document.body);
			this._proxy = new WCF.Action.Proxy();

			this._didInit = true;
		},

		/**
		 * Prepares 'move thread' action.
		 *
		 * @param        array<integer>        threadIDs
		 * @param        object                callback
		 * @param        integer                boardID
		 * @param        string                environment
		 */
		prepare: function (threadIDs, callback, boardID, environment) {
			if (!$.isFunction(callback)) {
				console.debug("[WBB.Thread.MoveHandler] Given callback is invalid.");
				return;
			}
			this._callback = callback;
			this._boardID = boardID || 0;
			this._threadIDs = threadIDs;
			this._environment = environment;

			// initialize handler on first use
			if (!this._didInit) {
				this._init();
			}

			this._proxy.setOption('data', {
				actionName: 'prepareMove',
				className: 'wbb\\data\\thread\\ThreadAction',
				objectIDs: this._threadIDs,
				parameters: {
					boardID: this._boardID
				}
			});
			this._proxy.setOption('success', $.proxy(this._success, this));
			this._proxy.sendRequest();
		},

		/**
		 * Handles successful AJAX calls.
		 *
		 * @param        object                data
		 * @param        string                textStatus
		 * @param        jQuery                jqXHR
		 */
		_success: function (data, textStatus, jqXHR) {
			this._dialog.data('objectIDs', data.objectIDs).html(data.returnValues.template);
			this._dialog.wcfDialog({
				title: WCF.Language.get('wbb.thread.edit.moveThreads')
			});

			// listen for submit event
			this._dialog.find('.formSubmit > input[type=submit]').click($.proxy(this._move, this));
		},

		/**
		 * Handles 'move thread' action.
		 *
		 * @param        object                event
		 */
		_move: function (event) {
			var $showMoveNotice = this._dialog.find('#showMoveNotice').prop('checked');
			var $boardID = parseInt(this._dialog.find('#boardID').prop('value'));

			if ($boardID) {
				// check if board id equals current board id
				if ($boardID == this._boardID && this._threadIDs.length == 1 && this._environment === 'thread') {
					this._dialog.find('#boardID').next('small.innerError').remove();
					$('<small class="innerError">' + WCF.Language.get('wbb.thread.edit.moveDestination.error.equalsOrigin') + '</small>').insertAfter(this._dialog.find('#boardID'));
					return;
				}

				this._proxy.setOption('data', {
					actionName: 'move',
					className: 'wbb\\data\\thread\\ThreadAction',
					objectIDs: this._dialog.data('objectIDs'),
					parameters: {
						boardID: $boardID,
						showMoveNotice: $showMoveNotice
					}
				});
				this._proxy.setOption('success', $.proxy(function (data) {
					this._callback(data);
				}, this));
				this._proxy.sendRequest();
			}

			// close dialog
			this._dialog.wcfDialog('close');
		},
	};

	/**
	 * Thread type chooser.
	 *
	 * @param        integer                type
	 */
	WBB.Thread.TypeChooser = Class.extend({
		/**
		 * board ids container
		 * @var        jQuery
		 */
		_boardIDsContainer: null,

		/**
		 * board type (0 = default, 1 = sticky, 2 = announcement)
		 * @var        integer
		 */
		_type: 0,

		/**
		 * Initializes the thread type chooser.
		 *
		 * @param        integer                type
		 */
		init: function (type) {
			this._type = parseInt(type);
			this._boardIDsContainer = $('#boardIDsContainer');

			var $inputElements = $('input[name=type]').click($.proxy(this._click, this));

			// set type on init
			var self = this;
			$inputElements.each(function (index, element) {
				var $element = $(element);

				if ($element.prop('value') == self._type) {
					$element.prop('checked', 'checked');
					self._update();

					return false;
				}
			});
		},

		/**
		 * Handles click event.
		 *
		 * @param        object                event
		 */
		_click: function (event) {
			this._type = parseInt($(event.currentTarget).prop('value'));

			this._update();
		},

		/**
		 * Updates board ids container state.
		 */
		_update: function () {
			switch (this._type) {
				/* default */
				case 0:
				/* sticky */
				case 1:
					this._boardIDsContainer.hide();
					break;

				/* announcement */
				case 2:
					this._boardIDsContainer.show();
					break;
			}
		}
	});
}
else {
	WBB.Thread.Editor = {
		_callbacks: {},
		_dialog: {},
		_dialogContainer: {},
		_environment: "",
		_notification: {},
		_proxy: {},
		_threadID: 0,
		beginEdit: function() {},
		registerCallback: function() {},
		_success: function() {},
		_showEditor: function() {},
		_save: function() {},
		_saveEdit: function() {},
		_getValues: function() {},
		_cancel: function() {},
		Default: function() {},
		LabelChooser: function() {},
		Language: function() {},
		TagList: function() {}
	};

	WBB.Thread.Editor.Default = Class.extend({
		init: function() {},
		_handleThreadType: function() {},
		_click: function() {},
		_update: function() {},
		getValues: function() {},
		saved: function() {},
		showErrors: function() {}
	});

	WBB.Thread.Editor.LabelChooser = WCF.Label.Chooser.extend({
		init: function() {},
		getValues: function() {},
		saved: function() {},
		showErrors: function() {},
		_container: {},
		_groups: {},
		_showWithoutSelection: false,
		_initContainers: function() {},
		_click: function() {},
		_selectLabel: function() {},
		_submit: function() {}
	});

	WBB.Thread.Editor.Language = WCF.Language.Chooser.extend({
		init: function() {},
		getValues: function() {},
		saved: function() {}
	});

	WBB.Thread.Editor.TagList = Class.extend({
		_api: {},
		init: function() {},
		getValues: function() {},
		saved: function() {}
	});

	WBB.Thread.InlineEditor = WCF.InlineEditor.extend({
		_boardID: 0,
		_environment: "",
		_permissions: {},
		_redirectURL: "",
		_updateHandler: {},
		_setOptions: function() {},
		setUpdateHandler: function() {},
		_getTriggerElement: function() {},
		_validate: function() {},
		_execute: function() {},
		_updateThread: function() {},
		_updateState: function() {},
		_getPermission: function() {},
		setEnvironment: function() {},
		setPermission: function() {},
		setPermissions: function() {},
		_callbacks: {},
		_dropdowns: {},
		_elements: {},
		_notification: {},
		_options: {},
		_proxy: {},
		_triggerElements: {},
		_updateData: {},
		init: function() {},
		_closeAll: function() {},
		registerCallback: function() {},
		_show: function() {},
		_validateCallbacks: function() {},
		_success: function() {},
		_click: function() {},
		_executeCallback: function() {},
		_hide: function() {}
	});

	WBB.Thread.Clipboard = Class.extend({
		_boardID: 0,
		_environment: "",
		_updateHandler: {},
		init: function() {},
		_clipboardAction: function() {},
		_move: function() {},
		_evaluateResponse: function() {}
	});

	WBB.Thread.UpdateHandler = Class.extend({
		_markAsDoneHandler: {},
		_threads: {},
		init: function() {},
		setMarkAsDoneHandler: function() {},
		update: function() {},
		_updateProperty: function() {},
		_handleCustomProperty: function() {},
		_close: function() {},
		_delete: function() {},
		_deleteNote: function() {},
		_disable: function() {},
		_done: function() {},
		_enable: function() {},
		_moved: function() {},
		_open: function() {},
		_restore: function() {},
		_scrape: function() {},
		_sticky: function() {},
		_trash: function() {},
		_undone: function() {},
		_updateTitle: function() {},
		_getCloseIcon: function() {},
		_getDoneIcon: function() {},
		_getMoveIcon: function() {},
		_getUndoneIcon: function() {},
		getValue: function() {}
	});

	WBB.Thread.UpdateHandler.Board = WBB.Thread.UpdateHandler.extend({
		_close: function() {},
		_delete: function() {},
		_deleteNote: function() {},
		_disable: function() {},
		_done: function() {},
		_enable: function() {},
		_moved: function() {},
		_showMoveNotice: function() {},
		_open: function() {},
		_restore: function() {},
		_trash: function() {},
		_undone: function() {},
		_markAsDoneHandler: {},
		_threads: {},
		init: function() {},
		setMarkAsDoneHandler: function() {},
		update: function() {},
		_updateProperty: function() {},
		_handleCustomProperty: function() {},
		_scrape: function() {},
		_sticky: function() {},
		_updateTitle: function() {},
		_getCloseIcon: function() {},
		_getDoneIcon: function() {},
		_getMoveIcon: function() {},
		_getUndoneIcon: function() {},
		getValue: function() {}
	});

	WBB.Thread.UpdateHandler.Thread = WBB.Thread.UpdateHandler.extend({
		_boardID: 0,
		_postManager: {},
		init: function() {},
		setPostHandler: function() {},
		update: function() {},
		_close: function() {},
		_delete: function() {},
		_disable: function() {},
		_done: function() {},
		_enable: function() {},
		_moved: function() {},
		_showMoveNotice: function() {},
		_open: function() {},
		_scrape: function() {},
		_sticky: function() {},
		_restore: function() {},
		_trash: function() {},
		_undone: function() {},
		_markAsDoneHandler: {},
		_threads: {},
		setMarkAsDoneHandler: function() {},
		_updateProperty: function() {},
		_handleCustomProperty: function() {},
		_deleteNote: function() {},
		_updateTitle: function() {},
		_getCloseIcon: function() {},
		_getDoneIcon: function() {},
		_getMoveIcon: function() {},
		_getUndoneIcon: function() {},
		getValue: function() {}
	});

	WBB.Thread.MergeHandler = Class.extend({
		_className: "",
		_dialog: {},
		_dialogTitle: "",
		_objectIDs: {},
		_proxy: {},
		_successMessage: "",
		init: function() {},
		load: function() {},
		_success: function() {},
		_submit: function() {},
		_getParameters: function() {}
	});

	WBB.Thread.AssignLabelHandler = {
		_boardID: 0,
		_dialog: {},
		_objectIDs: {},
		prepare: function() {},
		_click: function() {},
		_success: function() {}
	};

	WBB.Thread.CopyHandler = {
		_boardID: 0,
		_i: 0,
		_objectIDs: {},
		prepare: function() {},
		copy: function() {}
	};

	WBB.Thread.MoveHandler = {
		_boardID: 0,
		_callback: {},
		_dialog: {},
		_didInit: false,
		_environment: "",
		_proxy: {},
		_threadIDs: {},
		_init: function() {},
		prepare: function() {},
		_success: function() {},
		_move: function() {}
	};

	WBB.Thread.TypeChooser = {
		_boardIDsContainer: {},
		_type: 0,
		init: function() {},
		_click: function() {},
		_update: function() {}
	};
}

/**
 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Thread/LastPageHandler` instead
 */
WBB.Thread.LastPageHandler = Class.extend({
	init: function(_threadID, lastPostTime, pageNo) {
		// Calls to this method imply that the template is outdated, therefore
		// we need to manually set the new data attributes to keep it in sync.
		const threadPostList = document.querySelector(".wbbThreadPostList");
		threadPostList.dataset.lastPostTime = lastPostTime.toString();
		threadPostList.dataset.pageNo = pageNo.toString();

		// The `WBB.Thread.LastPageHandler` was only called when the current
		// page is the last page and the sort order is ascending.
		threadPostList.dataset.isLastPage = "true";
		threadPostList.dataset.sortOrder = "ASC";

		require(["WoltLabSuite/Forum/Ui/Thread/LastPageHandler"], ({ LastPageHandler }) => {
			LastPageHandler.setup();
		});
	}
});

if (COMPILER_TARGET_DEFAULT) {
	/**
	 * Provides a toggle handler for mark as done.
	 *
	 * @param        WBB.Thread.UpdateHandler.Board                updateHandler
	 */
	WBB.Thread.MarkAsDone = Class.extend({
		/**
		 * action proxy
		 * @var        WCF.Action.Proxy
		 */
		_proxy: null,

		/**
		 * thread update handler
		 * @var        WBB.Thread.UpdateHandler.Board
		 */
		_updateHandler: null,

		/**
		 * list of threads
		 * @var        object
		 */
		_threads: {},

		/**
		 * Initializes the mark as done handler.
		 *
		 * @param        WBB.Thread.UpdateHandler.Board                updateHandler
		 */
		init: function (updateHandler) {
			this._updateHandler = updateHandler;
			this._threads = {};

			this._proxy = new WCF.Action.Proxy({
				success: $.proxy(this._success, this)
			});

			var self = this;
			$('.wbbThread').each(function (index, thread) {
				var $thread = $(thread);
				var $threadID = $thread.data('threadID');

				self._threads[$threadID] = $thread;

				if ($thread.data('canMarkAsDone')) {
					self.watch($threadID);
				}
			});

			// register with update handler
			this._updateHandler.setMarkAsDoneHandler(this);
		},

		/**
		 * Watches for double clicks on "mark as done" icon.
		 *
		 * @param        integer                threadID
		 */
		watch: function (threadID) {
			if (this._threads[threadID] && this._threads[threadID].data('canMarkAsDone')) {
				this._threads[threadID].find('.jsMarkAsDone').data('threadID', threadID).dblclick($.proxy(this._dblclick, this));
			}
		},

		/**
		 * Handles double click on the "mark as done" icon.
		 *
		 * @param        object                event
		 */
		_dblclick: function (event) {
			var $icon = $(event.currentTarget);
			if ($icon[0].nodeName == 'LI') var $isDone = ($icon.find('.fa-check-square-o').length ? true : false);
			else var $isDone = ($icon.hasClass('fa-check-square-o') ? true : false);

			this._proxy.setOption('data', {
				actionName: ($isDone ? 'un' : '') + 'done',
				className: 'wbb\\data\\thread\\ThreadAction',
				objectIDs: [$icon.data('threadID')]
			});
			this._proxy.sendRequest();
		},

		/**
		 * Handles successful AJAX requests.
		 *
		 * @param        object                data
		 * @param        string                textStatus
		 * @param        jQuery                jqXHR
		 */
		_success: function (data, textStatus, jqXHR) {
			for (var $threadID in data.returnValues.threadData) {
				this._updateHandler.update($threadID, data.returnValues.threadData[$threadID]);
			}
		}
	});

	/**
	 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Thread/SimilarThreads` instead
	 */
	WBB.Thread.SimilarThreads = Class.extend({
		init: function (element, boardID) {
			require(["WoltLabSuite/Forum/Ui/Thread/SimilarThreads"], ({ SimilarThreads }) => {
				new SimilarThreads(element, boardID);
			});
		}
	});

	/**
	 * @deprecated	5.5, use `WoltLabSuite/Forum/Controller/Thread/WatchedList` instead
	 */
	WBB.Thread.WatchedThreadList = Class.extend({
		init: function () {
			require(["WoltLabSuite/Forum/Controller/Thread/WatchedList"], (ControllerThreadWatchedList) => {
				ControllerThreadWatchedList.setup();
			});
		}
	});
}
else {
	WBB.Thread.MarkAsDone = Class.extend({
		_proxy: {},
		_updateHandler: {},
		_threads: {},
		init: function() {},
		watch: function() {},
		_dblclick: function() {},
		_success: function() {}
	});

	WBB.Thread.SimilarThreads = Class.extend({
		_proxy: {},
		_element: {},
		_boardID: 0,
		init: function() {},
		_load: function() {},
		_success: function() {}
	});

	WBB.Thread.WatchedThreadList = Class.extend({
		_button: {},
		_markAllCheckbox: {},
		init: function() {},
		_mark: function() {},
		_markAll: function() {},
		_stopWatching: function() {},
		_updateButtonLabel: function() {}
	});
}

/**
 * Namespace for post related actions.
 */
WBB.Post = { };

/**
 * Namespace for post add related actions.
 */
WBB.Post.Add = { };

if (COMPILER_TARGET_DEFAULT) {
	/**
	 * Manages editor message options.
	 */
	WBB.Post.Add.MessageOptions = Class.extend({
		/**
		 * subscription button
		 * @var        jQuery
		 */
		_subscribeButton: null,

		/**
		 * Initializes the WBB.Post.Add.MessageOptions object.
		 */
		init: function () {
			this._subscribeButton = $('.jsSubscribeButton');

			WCF.System.Event.addListener('com.woltlab.wcf.redactor', 'updateMessageOptions', $.proxy(this._updateMessageOptions, this));
			WCF.System.Event.addListener('com.woltlab.wcf.objectWatch', 'updatedSubscription', $.proxy(this._updateMessageOptions, this));
		},

		/**
		 * Updates editor message options.
		 *
		 * @param        object                data
		 */
		_updateMessageOptions: function (data) {
			var $checkbox = $('#settings_text').find('input[name=subscribeThread]');
			if ($checkbox.length) {
				if (this._subscribeButton.attr("data-is-subscribed") === "1") {
					$checkbox.prop('checked', 'checked');
				}
				else {
					$checkbox.prop('checked', false);
				}
			}
		}
	});

	/**
	 * Provides extended actions for post clipboard actions.
	 */
	WBB.Post.Clipboard = Class.extend({
		/**
		 * post manager
		 * @var        {WoltLabSuite/Forum/Ui/Post/Manager}
		 */
		_postManager: null,

		/**
		 * thread update handler
		 * @var        WBB.Thread.UpdateHandler
		 */
		_threadUpdateHandler: null,

		/**
		 * Initializes a new WBB.Post.Clipboard object.
		 *
		 * @param        {WoltLabSuite/Forum/Ui/Post/Manager}        postManager
		 */
		init: function (postManager) {
			this._postManager = postManager;

			WCF.System.Event.addListener('com.woltlab.wcf.clipboard', 'com.woltlab.wbb.post', (function (data) {
				if (data.responseData === null) {
					this._execute(data.data.actionName, data.data.parameters);
				}
				else {
					this._evaluateResponse(data.data.actionName, data.responseData);
				}
			}).bind(this));
		},

		/**
		 * Sets the thread update handler object.
		 *
		 * @param        {WBB.Thread.UpdateHandler}      threadUpdateHandler
		 */
		setThreadUpdateHandler: function (threadUpdateHandler) {
			this._threadUpdateHandler = threadUpdateHandler;
		},

		/**
		 * Handles clipboard actions.
		 *
		 * @param        {string}        actionName
		 * @param        {Object}        parameters
		 */
		_execute: function (actionName, parameters) {
			switch (actionName) {
				case 'com.woltlab.wbb.post.copyToExistingThread':
					WBB.Post.CopyToExistingThread.prepare(parameters);
					break;

				case 'com.woltlab.wbb.post.copyToNewThread':
					WBB.Post.CopyToNewThread.prepare(parameters);
					break;

				case 'com.woltlab.wbb.post.merge':
					var $mergeHandler = new WBB.Post.MergeHandler(parameters.objectIDs);
					$mergeHandler.load();
					break;

				case 'com.woltlab.wbb.post.moveToNewThread':
					WBB.Post.MoveToNewThread.prepare(parameters);
					break;
			}
		},

		/**
		 * Evaluates AJAX responses.
		 *
		 * @param        {string}        actionName
		 * @param        {Object}        data
		 */
		_evaluateResponse: function (actionName, data) {

			if (actionName === 'com.woltlab.wbb.post.moveToExistingThread') {
				var $notification = new WCF.System.Notification();
				$notification.show(function () {
					window.location = data.returnValues.redirectURL;
				});

				return;
			}

			if (data.returnValues.postData && $.getLength(data.returnValues.postData)) {
				this._postManager._ajaxSuccess(data);
			}

			if (this._threadUpdateHandler && data.returnValues.threadData && $.getLength(data.returnValues.threadData)) {
				for (var threadId in data.returnValues.threadData) {
					if (data.returnValues.threadData.hasOwnProperty(threadId)) {
						this._threadUpdateHandler.update(threadId, data.returnValues.threadData[threadId]);
					}
				}
			}
		}
	});

	/**
	 * Merges multiple posts.
	 *
	 * @see        WBB.Thread.MergeHandler
	 */
	WBB.Post.MergeHandler = WBB.Thread.MergeHandler.extend({
		/**
		 * @see        WBB.Thread.MergeHandler.init()
		 */
		init: function (objectIDs) {
			this._super(objectIDs);

			this._className = 'wbb\\data\\post\\PostAction';
			this._dialogTitle = WCF.Language.get('wbb.post.edit.merge');
			this._successMessage = WCF.Language.get('wbb.post.edit.merge.success');
		},

		/**
		 * @see        WBB.Thread.MergeHandler._success()
		 */
		_success: function (data, textStatus, jqXHR) {
			this._super(data, textStatus, jqXHR);

			if (data.returnValues.redirectURL === undefined) {
				this._dialog.find('.formSubmit > button[data-type=submit]').enable();
			}
		},

		/**
		 * @see        WBB.Thread.MergeHandler._getParameters()
		 */
		_getParameters: function () {
			var $postOrder = [];
			this._dialog.find('.jsPostContainer').each(function (index, container) {
				$postOrder.push($(container).data('postID'));
			});

			return {
				postOrder: $postOrder
			};
		}
	});

	/**
	 * Copies selected posts into an existing thread.
	 */
	WBB.Post.CopyToExistingThread = {
		/**
		 * Prepares copying of posts.
		 *
		 * @param        object                parameters
		 */
		prepare: function (parameters) {
			new WCF.System.Worker('copyToExistingThread', 'wbb\\data\\post\\PostAction', WCF.Language.get('wbb.post.copy.title'), parameters);
		}
	};

	/**
	 * Copies selected posts into a new thread.
	 */
	WBB.Post.CopyToNewThread = {
		/**
		 * board select element
		 * @var        jQuery
		 */
		_boardSelect: null,

		/**
		 * dialog overlay
		 * @var        jQuery
		 */
		_dialog: null,

		/**
		 * list of affected post ids
		 * @var        array<integer>
		 */
		_objectIDs: [],

		/**
		 * submit button
		 * @var        jQuery
		 */
		_submitButton: null,

		/**
		 * topic input element
		 * @var        jQuery
		 */
		_topic: null,

		/**
		 * Prepares copying of posts.
		 *
		 * @param        object                parameters
		 */
		prepare: function (parameters) {
			if (this._dialog === null) {
				this._dialog = $('<div>' + parameters.template + '</div>').hide().appendTo(document.body);
				this._dialog.wcfDialog({
					title: WCF.Language.get('wbb.post.copy.title')
				});
			}
			else {
				this._dialog.html($.parseHTML(parameters.template)).wcfDialog('open');
			}

			this._boardSelect = $('#postCopyNewThreadBoardID');
			this._objectIDs = parameters.objectIDs;
			this._submitButton = this._dialog.find('.formSubmit > button[data-type=submit]').disable().click($.proxy(this._submit, this));
			this._topic = $('#postCopyNewThreadTopic').keyup($.proxy(this._keyUp, this)).focus();
			if ($.trim(this._topic.val()).length) {
				this._submitButton.enable();
			}

			if ($.browser.mozilla && $.browser.touch) {
				this._topic.on('input', $.proxy(this._keyUp, this));
			}
		},

		/**
		 * Handles the 'keyup' event to enable/disable the submit button.
		 */
		_keyUp: function () {
			if ($.trim(this._topic.val()).length > 0) {
				this._submitButton.enable();
			}
			else {
				this._submitButton.disable();
			}
		},

		/**
		 * Submits the form input elements.
		 */
		_submit: function () {
			this._dialog.wcfDialog('close');

			new WCF.System.Worker('copyToNewThread', 'wbb\\data\\post\\PostAction', WCF.Language.get('wbb.post.copy.title'), {
				boardID: parseInt(this._boardSelect.val()),
				objectIDs: this._objectIDs,
				topic: $.trim(this._topic.val())
			});
		}
	};

	/**
	 * Moves selected posts into a new thread.
	 */
	WBB.Post.MoveToNewThread = {
		/**
		 * board select element
		 * @var        jQuery
		 */
		_boardSelect: null,

		/**
		 * dialog overlay
		 * @var        jQuery
		 */
		_dialog: null,

		/**
		 * list of affected post ids
		 * @var        array<integer>
		 */
		_objectIDs: [],

		/**
		 * submit button
		 * @var        jQuery
		 */
		_submitButton: null,

		/**
		 * topic input element
		 * @var        jQuery
		 */
		_topic: null,

		/**
		 * Prepares moving of posts.
		 *
		 * @param        object                parameters
		 */
		prepare: function (parameters) {
			if (this._dialog === null) {
				this._dialog = $('<div>' + parameters.template + '</div>').hide().appendTo(document.body);
				this._dialog.wcfDialog({
					title: WCF.Language.get('wbb.post.moveToNewThread')
				});
			}
			else {
				this._dialog.html($.parseHTML(parameters.template)).wcfDialog('open');
			}

			this._boardSelect = $('#postMoveNewThreadBoardID');
			this._objectIDs = parameters.objectIDs;
			this._submitButton = this._dialog.find('.formSubmit > button[data-type=submit]').disable().click($.proxy(this._submit, this));
			this._topic = $('#postMoveNewThreadTopic').keyup($.proxy(this._keyUp, this)).focus();
			if ($.trim(this._topic.val()).length) {
				this._submitButton.enable();
			}

			if ($.browser.mozilla && $.browser.touch) {
				this._topic.on('input', $.proxy(this._keyUp, this));
			}
		},

		/**
		 * Handles the 'keyup' event to enable/disable the submit button.
		 */
		_keyUp: function () {
			if ($.trim(this._topic.val()).length > 0) {
				this._submitButton.enable();
			}
			else {
				this._submitButton.disable();
			}
		},

		/**
		 * Submits the form input elements.
		 */
		_submit: function () {
			this._submitButton.disable();

			new WCF.Action.Proxy({
				autoSend: true,
				data: {
					actionName: 'moveToNewThread',
					className: 'wbb\\data\\post\\PostAction',
					objectIDs: this._objectIDs,
					parameters: {
						boardID: parseInt(this._boardSelect.val()),
						topic: $.trim(this._topic.val())
					}
				},
				success: $.proxy(this._success, this)
			});
		},

		/**
		 * Handles successful AJAX requests.
		 *
		 * @param        object                data
		 * @param        string                textStatus
		 * @param        jQuery                jqXHR
		 */
		_success: function (data, textStatus, jqXHR) {
			var $notification = new WCF.System.Notification();
			$notification.show(function () {
				window.location = data.returnValues.redirectURL;
			});
		}
	};
}
else {
	WBB.Post.Add.MessageOptions = Class.extend({
		_subscribeButton: {},
		init: function() {},
		_updateMessageOptions: function() {}
	});

	WBB.Post.Clipboard = Class.extend({
		_postManager: {},
		_threadUpdateHandler: {},
		init: function() {},
		setThreadUpdateHandler: function() {},
		_execute: function() {},
		_evaluateResponse: function() {}
	});

	WBB.Post.MergeHandler = WBB.Thread.MergeHandler.extend({
		init: function() {},
		_success: function() {},
		_getParameters: function() {},
		_className: "",
		_dialog: {},
		_dialogTitle: "",
		_objectIDs: {},
		_proxy: {},
		_successMessage: "",
		load: function() {},
		_submit: function() {}
	});

	WBB.Post.CopyToExistingThread = {
		prepare: function() {}
	};

	WBB.Post.CopyToNewThread = {
		_boardSelect: {},
		_dialog: {},
		_objectIDs: {},
		_submitButton: {},
		_topic: {},
		prepare: function() {},
		_keyUp: function() {},
		_submit: function() {}
	};

	WBB.Post.MoveToNewThread = {
		_boardSelect: {},
		_dialog: {},
		_objectIDs: {},
		_submitButton: {},
		_topic: {},
		prepare: function() {},
		_keyUp: function() {},
		_submit: function() {},
		_success: function() {}
	};
}

/**
 * Provides a flexible post preview.
 *
 * @see	WCF.Popover
 * @deprecated  5.3     handled via `WoltLabSuite/Core/Controller/Popover`
 */
WBB.Post.Preview = WCF.Popover.extend({
	/**
	 * action proxy
	 * @var	WCF.Action.Proxy
	 */
	_proxy: null,

	/**
	 * @see	WCF.Popover.init()
	 */
	init: function() {
		this._super('.wbbTopicLink, .wbbPostLink');

		// init proxy
		this._proxy = new WCF.Action.Proxy({
			showLoadingOverlay: false
		});

		WCF.DOMNodeInsertedHandler.addCallback('WBB.Post.Preview', $.proxy(this._initContainers, this));
	},

	/**
	 * @see	WCF.Popover._loadContent()
	 */
	_loadContent: function() {
		var $link = $('#' + this._activeElementID);

		if ($link.hasClass('wbbTopicLink')) {
			this._proxy.setOption('data', {
				actionName: 'getPostPreview',
				className: 'wbb\\data\\thread\\ThreadAction',
				objectIDs: [ $link.data('threadID') ],
				parameters: {
					sortOrder: ($link.data('sortOrder') ? $link.data('sortOrder') : 'ASC')
				}
			});
		}
		else {
			this._proxy.setOption('data', {
				actionName: 'getPostPreview',
				className: 'wbb\\data\\post\\PostAction',
				objectIDs: [ $link.data('postID') ]
			});
		}

		var $elementID = this._activeElementID;
		var self = this;
		this._proxy.setOption('success', function(data, textStatus, jqXHR) {
			self._insertContent($elementID, data.returnValues.template, true);
		});
		this._proxy.setOption('failure', function(data, jqXHR, textStatus, errorThrown) {
			self._insertContent($elementID, data.message, true);

			return false;
		});
		this._proxy.sendRequest();
	}
});

if (COMPILER_TARGET_DEFAULT) {
	/**
	 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Post/IpAddressHandler` instead
	 */
	WBB.Post.IPAddressHandler = Class.extend({
		init: function () {
			require(["WoltLabSuite/Forum/Ui/Post/IpAddressHandler"], (IpAddressHandler) => {
				new IpAddressHandler();
			});
		},
	});

	/**
	 * @deprecated	5.5, use `WoltLabSuite/Forum/Ui/Post/Quote` instead
	 */
	WBB.Post.QuoteHandler = WCF.Message.Quote.Handler.extend({
		init: function (quoteManager) {
			require(["WoltLabSuite/Forum/Ui/Post/Quote"], ({ UiPostQuote }) => {
				new UiPostQuote(quoteManager);
			});
		}
	});
}
else {
	WBB.Post.IPAddressHandler = Class.extend({
		_cache: {},
		_dialog: {},
		_proxy: {},
		init: function() {},
		_initButtons: function() {},
		_click: function() {},
		_success: function() {},
		_showDialog: function() {}
	});

	WBB.Post.QuoteHandler = WCF.Message.Quote.Handler.extend({
		init: function() {},
		_activeContainerID: "",
		_className: "",
		_containers: {},
		_containerSelector: "",
		_copyQuote: {},
		_message: "",
		_messageBodySelector: "",
		_objectID: 0,
		_objectType: "",
		_proxy: {},
		_quoteManager: {},
		_initContainers: function() {},
		_mouseDown: function() {},
		_getNodeText: function() {},
		_mouseUp: function() {},
		_normalize: function() {},
		_getBoundingRectangle: function() {},
		_initCopyQuote: function() {},
		_getSelectedText: function() {},
		_saveFullQuote: function() {},
		_saveQuote: function() {},
		_saveAndInsertQuote: function() {},
		_success: function() {},
		updateFullQuoteObjectIDs: function() {}
	});
}
