<?php

namespace wbb\acp\form;

use wbb\data\board\Board;
use wbb\data\board\BoardAction;
use wbb\data\board\BoardCache;
use wbb\data\board\RealtimeBoardNodeList;
use wcf\data\package\PackageCache;
use wcf\form\AbstractForm;
use wcf\system\acl\ACLHandler;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\language\I18nHandler;
use wcf\system\style\StyleHandler;
use wcf\system\user\collapsible\content\UserCollapsibleContentHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\WCF;
use wcf\util\JSON;

/**
 * Shows the board edit form.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Acp\Form
 */
class BoardEditForm extends BoardAddForm
{
    /**
     * @inheritDoc
     */
    public $templateName = 'boardAdd';

    /**
     * @inheritDoc
     */
    public $activeMenuItem = 'wbb.acp.menu.link.board.list';

    /**
     * @inheritDoc
     */
    public $neededPermissions = ['admin.board.canEditBoard'];

    /**
     * id of the edited board
     * @var int
     */
    public $boardID = 0;

    /**
     * edited board object
     * @var Board
     */
    public $board;

    /**
     * @inheritDoc
     */
    public function readParameters()
    {
        parent::readParameters();

        if (isset($_REQUEST['id'])) {
            $this->boardID = \intval($_REQUEST['id']);
        }
        $this->board = new Board($this->boardID);
        if (!$this->board->boardID) {
            throw new PermissionDeniedException();
        }
    }

    /**
     * @inheritDoc
     */
    protected function validateParentID()
    {
        parent::validateParentID();

        if ($this->parentID) {
            if ($this->boardID == $this->parentID) {
                throw new UserInputException('parentID', 'invalid');
            }

            $board = BoardCache::getInstance()->getBoard($this->parentID);
            while ($board->parentID != null) {
                $board = BoardCache::getInstance()->getBoard($board->parentID);
                if ($board->boardID == $this->boardID) {
                    throw new UserInputException('parentID', 'invalid');
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        parent::readData();

        if (empty($_POST)) {
            I18nHandler::getInstance()->setOptions(
                'description',
                PackageCache::getInstance()->getPackageID('com.woltlab.wbb'),
                $this->board->description ?: '',
                'wbb.board.board\d+.description'
            );
            I18nHandler::getInstance()->setOptions(
                'metaDescription',
                PackageCache::getInstance()->getPackageID('com.woltlab.wbb'),
                $this->board->metaDescription ?: '',
                'wbb.board.board\d+.metaDescription'
            );
            I18nHandler::getInstance()->setOptions(
                'title',
                PackageCache::getInstance()->getPackageID('com.woltlab.wbb'),
                $this->board->title ?: '',
                'wbb.board.board\d+'
            );

            $this->boardType = $this->board->boardType;
            $this->countUserPosts = $this->board->countUserPosts;
            $this->daysPrune = $this->board->daysPrune;
            $this->description = $this->board->description;
            $this->descriptionUseHtml = $this->board->descriptionUseHtml;
            $this->enableMarkingAsDone = $this->board->enableMarkingAsDone;
            $this->enableBestAnswer = $this->board->enableBestAnswer;
            $this->ignorable = $this->board->ignorable;
            $this->isClosed = $this->board->isClosed;
            $this->isInvisible = $this->board->isInvisible;
            $this->isPrivate = $this->board->isPrivate;
            $this->parentID = $this->board->parentID;
            $this->position = $this->board->position;
            $this->postSortOrder = $this->board->postSortOrder;
            $this->postsPerPage = $this->board->postsPerPage;
            $this->searchable = $this->board->searchable;
            $this->searchableForSimilarThreads = $this->board->searchableForSimilarThreads;
            $this->showSubBoards = $this->board->showSubBoards;
            $this->sortField = $this->board->sortField;
            $this->sortOrder = $this->board->sortOrder;
            $this->threadsPerPage = $this->board->threadsPerPage;
            $this->title = $this->board->title;
            $this->styleID = $this->board->styleID;
            $this->externalURL = $this->board->externalURL;
            $this->metaDescription = $this->board->metaDescription;
            $this->formID = $this->board->formID;

            foreach ($this->board->getIconData() as $type => $data) {
                foreach ($data as $key => $value) {
                    $this->iconData[$type][$key] = $value;
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function save()
    {
        AbstractForm::save();

        // update description
        $this->description = 'wbb.board.board' . $this->board->boardID . '.description';
        if (I18nHandler::getInstance()->isPlainValue('description')) {
            I18nHandler::getInstance()->remove($this->description);
            $this->description = I18nHandler::getInstance()->getValue('description');
        } else {
            I18nHandler::getInstance()->save(
                'description',
                $this->description,
                'wbb.board',
                PackageCache::getInstance()->getPackageID('com.woltlab.wbb')
            );
        }

        $this->metaDescription = 'wbb.board.board' . $this->board->boardID . '.metaDescription';
        if (I18nHandler::getInstance()->isPlainValue('metaDescription')) {
            I18nHandler::getInstance()->remove($this->metaDescription);
            $this->metaDescription = I18nHandler::getInstance()->getValue('metaDescription');
        } else {
            I18nHandler::getInstance()->save(
                'metaDescription',
                $this->metaDescription,
                'wbb.board',
                PackageCache::getInstance()->getPackageID('com.woltlab.wbb')
            );
        }

        // update title
        $this->title = 'wbb.board.board' . $this->board->boardID;
        if (I18nHandler::getInstance()->isPlainValue('title')) {
            I18nHandler::getInstance()->remove($this->title);
            $this->title = I18nHandler::getInstance()->getValue('title');
        } else {
            I18nHandler::getInstance()->save(
                'title',
                $this->title,
                'wbb.board',
                PackageCache::getInstance()->getPackageID('com.woltlab.wbb')
            );
        }

        // build array with properties depending on the board type;
        // initialize it with default values
        $properties = [
            'countUserPosts' => 1,
            'daysPrune' => 0,
            'enableMarkingAsDone' => 0,
            'ignorable' => 1,
            'isClosed' => 0,
            'isPrivate' => 0,
            'postSortOrder' => '',
            'postsPerPage' => 0,
            'searchable' => 1,
            'searchableForSimilarThreads' => 1,
            'showSubBoards' => 1,
            'sortField' => '',
            'sortOrder' => '',
            'styleID' => null,
            'threadsPerPage' => 0,
            'formID' => null,
        ];

        switch ($this->boardType) {
            case Board::TYPE_BOARD:
                $properties = [
                    'countUserPosts' => $this->countUserPosts ? 1 : 0,
                    'daysPrune' => $this->daysPrune,
                    'enableMarkingAsDone' => $this->enableMarkingAsDone ? 1 : 0,
                    'enableBestAnswer' => $this->enableBestAnswer ? 1 : 0,
                    'ignorable' => $this->ignorable ? 1 : 0,
                    'isClosed' => $this->isClosed ? 1 : 0,
                    'isPrivate' => $this->isPrivate ? 1 : 0,
                    'postSortOrder' => $this->postSortOrder,
                    'postsPerPage' => $this->postsPerPage,
                    'searchable' => $this->searchable ? 1 : 0,
                    'searchableForSimilarThreads' => $this->searchableForSimilarThreads ? 1 : 0,
                    'showSubBoards' => $this->showSubBoards ? 1 : 0,
                    'sortField' => $this->sortField,
                    'sortOrder' => $this->sortOrder,
                    'threadsPerPage' => $this->threadsPerPage,
                    'formID' => $this->formID ?: null,
                ];
                // no break
            case Board::TYPE_CATEGORY:
                $properties['styleID'] = $this->styleID ?: null;
                break;
        }

        $iconData = JSON::encode($this->getIconData());

        // update board
        $this->objectAction = new BoardAction(
            [$this->board],
            'update',
            [
                'data' => \array_merge($this->additionalFields, [
                    'boardType' => $this->boardType,
                    'description' => $this->description,
                    'descriptionUseHtml' => $this->descriptionUseHtml ? 1 : 0,
                    'isInvisible' => $this->isInvisible ? 1 : 0,
                    'parentID' => $this->parentID ?: null,
                    'position' => $this->position,
                    'title' => $this->title,
                    'externalURL' => $this->externalURL,
                    'iconData' => $iconData,
                    'metaDescription' => $this->metaDescription,
                ], $properties),
            ]
        );
        $this->objectAction->executeAction();

        // save ACL
        ACLHandler::getInstance()->save($this->board->boardID, $this->objectTypeID);
        ACLHandler::getInstance()->disableAssignVariables();
        UserStorageHandler::getInstance()->resetAll('wbbBoardPermissions');
        $this->saved();

        // check if board cannot be ignored anymore and remove it from
        // users' list of ignored boards
        if ($this->board->ignorable && !$this->ignorable) {
            UserCollapsibleContentHandler::getInstance()
                ->resetAll('com.woltlab.wbb.ignoredBoard', $this->board->boardID);
        }

        if ($iconData !== JSON::encode($this->board->getIconData())) {
            StyleHandler::resetStylesheets(false);
        }

        // show success message
        WCF::getTPL()->assign('success', true);
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        I18nHandler::getInstance()->assignVariables(!empty($_POST));

        $availableBoards = new RealtimeBoardNodeList();
        $availableBoards->readNodeTree();
        WCF::getTPL()->assign([
            'action' => 'edit',
            'boardID' => $this->boardID,
            'board' => $this->board,
            'availableBoards' => $availableBoards->getNodeList(),
        ]);
    }
}
