<?php

namespace wbb\data\post;

use wbb\data\thread\Thread;
use wbb\data\thread\ThreadEditor;
use wcf\data\like\Like;
use wcf\data\like\object\AbstractLikeObject;
use wcf\data\reaction\object\IReactionObject;
use wcf\system\user\notification\object\LikeUserNotificationObject;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\WCF;

/**
 * Likeable object implementation for posts.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Post
 *
 * @method  Post    getDecoratedObject()
 * @mixin   Post
 */
class LikeablePost extends AbstractLikeObject implements IReactionObject
{
    /**
     * @inheritDoc
     */
    protected static $baseClass = Post::class;

    /**
     * @inheritDoc
     */
    public function getTitle()
    {
        return $this->getDecoratedObject()->getTitle();
    }

    /**
     * @inheritDoc
     */
    public function getURL()
    {
        return $this->getDecoratedObject()->getLink();
    }

    /**
     * @inheritDoc
     */
    public function getUserID()
    {
        return $this->userID;
    }

    /**
     * @inheritDoc
     */
    public function updateLikeCounter($cumulativeLikes)
    {
        // update cumulative likes
        $postEditor = new PostEditor($this->getDecoratedObject());
        $postEditor->update(['cumulativeLikes' => $cumulativeLikes]);

        // check if this is the first post
        $thread = new Thread($this->getDecoratedObject()->threadID);
        if ($thread->firstPostID == $this->postID) {
            $threadEditor = new ThreadEditor($thread);
            $threadEditor->update(['cumulativeLikes' => $cumulativeLikes]);
        }
    }

    /**
     * @inheritDoc
     */
    public function sendNotification(Like $like)
    {
        if ($this->getDecoratedObject()->userID != WCF::getUser()->userID) {
            $notificationObject = new LikeUserNotificationObject($like);
            UserNotificationHandler::getInstance()->fireEvent(
                'like',
                'com.woltlab.wbb.likeablePost.notification',
                $notificationObject,
                [$this->getDecoratedObject()->userID],
                [
                    'objectID' => $this->getDecoratedObject()->postID,
                    'threadID' => $this->getDecoratedObject()->threadID,
                ],
                $this->getDecoratedObject()->postID
            );
        }
    }

    /**
     * @inheritDoc
     */
    public function getLanguageID()
    {
        return $this->getDecoratedObject()->getThread()->languageID;
    }
}
