<?php

namespace wbb\system\thread\status;

use InvalidArgumentException;
use wbb\data\post\Post;
use wbb\data\thread\ThreadList;
use wbb\system\board\status\BoardStatusHandler;
use wbb\system\user\notification\object\PostUserNotificationObject;
use wcf\data\user\User;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\WCF;

/**
 * Handles thread status related actions.
 *
 * @author  Joshua Ruesweg
 * @copyright   2001-2021 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Thread\Status
 * @since 5.5
 */
final class ThreadStatusHandler
{
    public const SUBSCRIPTION_MODE_NORMAL = 'normal';

    public const SUBSCRIPTION_MODE_WATCHING = 'watching';

    public const SUBSCRIPTION_MODE_IGNORING = 'ignoring';

    private static $statusCache = [];

    private static $statusThreadCache = [];

    /**
     * Returns the subscription status for a thread and a given user. If the `$user` is `null`,
     * the current user will be used.
     *
     * @throws InvalidArgumentException If the given user is a guest.
     */
    public static function getSubscriptionStatus(int $threadID, ?User $user = null): string
    {
        if ($user === null) {
            $user = WCF::getUser();
        }

        if (!$user->userID) {
            throw new \InvalidArgumentException("Guests does not have any status for threads.");
        }

        if (!isset(self::$statusCache[$user->userID][$threadID])) {
            $sql = "SELECT      status
                    FROM        wbb1_thread_user_status
                    WHERE       userID = ?
                            AND threadID = ?";
            $statement = WCF::getDB()->prepare($sql);
            $statement->execute([
                $user->userID,
                $threadID,
            ]);

            self::$statusCache[$user->userID][$threadID] = $statement->fetchSingleColumn() ?: self::SUBSCRIPTION_MODE_NORMAL;
        }

        return self::$statusCache[$user->userID][$threadID];
    }

    /**
     * Saves the given subscription status for a thread and a given user. If the `$user` is `null`,
     * the current user will be used.
     *
     * @throws InvalidArgumentException If the given user is a guest.
     */
    public static function saveSubscriptionStatus(array $threadIDs, string $status, ?User $user = null): void
    {
        if ($user === null) {
            $user = WCF::getUser();
        }

        if (!$user->userID) {
            throw new \InvalidArgumentException("Guests does not have any status for threads.");
        }

        switch ($status) {
            case self::SUBSCRIPTION_MODE_NORMAL:
                $conditionHandler = new PreparedStatementConditionBuilder();
                $conditionHandler->add("threadID IN (?)", [$threadIDs]);
                $conditionHandler->add("userID = ?", [$user->userID]);
                $sql = "DELETE FROM     wbb1_thread_user_status
                        {$conditionHandler}";
                $statement = WCF::getDB()->prepare($sql);
                $statement->execute($conditionHandler->getParameters());
                break;

            case self::SUBSCRIPTION_MODE_IGNORING:
            case self::SUBSCRIPTION_MODE_WATCHING:
                $sql = "INSERT INTO     wbb1_thread_user_status
                                        (threadID, userID, status)
                        VALUES          (?, ?, ?)
                        ON DUPLICATE KEY UPDATE status = VALUES(status)";
                $statement = WCF::getDB()->prepare($sql);
                foreach ($threadIDs as $threadID) {
                    $statement->execute([
                        $threadID,
                        $user->userID,
                        $status,
                    ]);
                }
                break;

            default:
                throw new \InvalidArgumentException(
                    \sprintf("Unknown thread status '%s' given.", $status)
                );
        }

        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbUnreadWatchedThreads');
        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbWatchedThreads');
        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbIgnoredThreads');

        foreach ($threadIDs as $threadID) {
            self::$statusCache[$user->userID][$threadID] = $status;
        }
    }

    /**
     * Deletes all subscriptions with the given status ('watching' or 'ignoring').
     *
     * @throws InvalidArgumentException If the status is not 'watching' nor 'ignoring'.
     */
    public static function deleteAllSubscriptions(array $threadIDs, string $status): void
    {
        if (
            !\in_array($status, [
                self::SUBSCRIPTION_MODE_WATCHING,
                self::SUBSCRIPTION_MODE_IGNORING,
            ])
        ) {
            throw new \InvalidArgumentException(\sprintf(
                "Unknown thread status '%s' given.",
                $status
            ));
        }

        $conditionBuilder = new PreparedStatementConditionBuilder();
        $conditionBuilder->add("threadID IN (?)", [$threadIDs]);
        $conditionBuilder->add("status = ?", [$status]);

        $sql = "DELETE FROM     wbb1_thread_user_status
                {$conditionBuilder}";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute($conditionBuilder->getParameters());

        UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedThreads');
        UserStorageHandler::getInstance()->resetAll('wbbWatchedThreads');
        UserStorageHandler::getInstance()->resetAll('wbbIgnoredThreads');
    }

    /**
     * Deletes all subscriptions with the given status ('watching' or 'ignoring') for a specific user.
     *
     * @throws InvalidArgumentException If the status is not 'watching' nor 'ignoring'.
     * @throws InvalidArgumentException If the given user is a guest.
     */
    public static function deleteAllSubscriptionsForUser(User $user, string $status): void
    {
        if (!$user->userID) {
            throw new \InvalidArgumentException(
                "Deleting status for guests is unsupported."
            );
        }

        if (
            !\in_array($status, [
                self::SUBSCRIPTION_MODE_WATCHING,
                self::SUBSCRIPTION_MODE_IGNORING,
            ])
        ) {
            throw new \InvalidArgumentException(\sprintf(
                "Unknown thread status '%s' given.",
                $status
            ));
        }

        $conditionBuilder = new PreparedStatementConditionBuilder();
        $conditionBuilder->add("userID = ?", [$user->userID]);
        $conditionBuilder->add("status = ?", [$status]);

        $sql = "DELETE FROM     wbb1_thread_user_status
                {$conditionBuilder}";
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute($conditionBuilder->getParameters());

        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbUnreadWatchedThreads');
        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbWatchedThreads');
        UserStorageHandler::getInstance()->reset([$user->userID], 'wbbIgnoredThreads');
    }

    /**
     * Returns a list of userIDs which have subscribed (watching) the given thread.
     *
     * @return int[]
     */
    public static function getSubscribers(int $threadID): array
    {
        return self::getUserIDsForStatus($threadID, self::SUBSCRIPTION_MODE_WATCHING);
    }

    /**
     * Appends a filter with the conidition builder to filter threads, which are ignored.
     */
    public static function addFilterForIgnoredThreads(ThreadList $threadList, ?User $user = null): void
    {
        if ($user === null) {
            $user = WCF::getUser();
        }

        if ($user->userID) {
            $threadList->getConditionBuilder()->add("threadID NOT IN (
                SELECT  threadID
                FROM    wbb" . \WCF_N . "_thread_user_status
                WHERE   status = ?
                    AND userID = ?
            )", [
                self::SUBSCRIPTION_MODE_IGNORING,
                $user->userID,
            ]);
        }
    }

    /**
     * Filters out the userIDs that ignore the given thread from the given userIDs and returns
     * only the userIDs that do not ignore the topic.
     *
     * @return int[]
     */
    public static function filterIgnoredUserIDs(array $userIDs, int $threadID): array
    {
        return \array_diff(
            $userIDs,
            self::getUserIDsForStatus($threadID, self::SUBSCRIPTION_MODE_IGNORING)
        );
    }

    /**
     * Fires the post notification for a specific post.
     */
    public static function firePostNotification(Post $post): void
    {
        $userIDs = self::getSubscribers($post->threadID);

        $userIDs = BoardStatusHandler::getInstance()->filterIgnoredUserIDs(
            $userIDs,
            $post->getThread()->boardID
        );

        $boardSubscriberUserIDs = BoardStatusHandler::getInstance()->getThreadAndPostSubscribers(
            $post->getThread()->boardID
        );
        if ($post->getThread()->languageID && !empty($boardSubscriberUserIDs)) {
            $boardSubscriberUserIDs = UserNotificationHandler::getInstance()->filterUsersByContentLanguage(
                $boardSubscriberUserIDs,
                $post->getThread()->languageID
            );
        }

        $userIDs = \array_unique(\array_merge(
            $userIDs,
            $boardSubscriberUserIDs
        ));

        $userIDs = self::filterIgnoredUserIDs($userIDs, $post->threadID);

        $userIDs = \array_diff(
            $userIDs,
            [$post->userID]
        );

        if (empty($userIDs)) {
            return;
        }

        UserNotificationHandler::getInstance()->fireEvent(
            'post',
            'com.woltlab.wbb.post',
            new PostUserNotificationObject($post),
            $userIDs
        );

        UserStorageHandler::getInstance()->reset($userIDs, 'wbbUnreadWatchedThreads');
    }

    private static function getUserIDsForStatus(int $threadID, string $status): array
    {
        if ($status === self::SUBSCRIPTION_MODE_NORMAL) {
            throw new \InvalidArgumentException("Fetching users with status 'normal' is not supported.");
        }

        if (!isset(self::$statusThreadCache[$threadID][$status])) {
            $sql = "SELECT      userID
                    FROM        wbb1_thread_user_status
                    WHERE       threadID = ?
                            AND status = ?";
            $statement = WCF::getDB()->prepare($sql);
            $statement->execute([
                $threadID,
                $status,
            ]);

            self::$statusThreadCache[$threadID][$status] = $statement->fetchAll(\PDO::FETCH_COLUMN);
        }

        return self::$statusThreadCache[$threadID][$status];
    }
}
