/**
 * Provides the thread form option dialog.
 *
 * @author  Joshua Ruesweg
 * @copyright  2001-2019 WoltLab GmbH
 * @license  WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module  WoltLabSuite/Forum/Acp/Ui/Board/Thread/Form/Option/DialogHandler
 * @since       5.2
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Ajax", "WoltLabSuite/Core/Core", "WoltLabSuite/Core/Event/Handler", "WoltLabSuite/Core/Language", "WoltLabSuite/Core/Language/Input", "WoltLabSuite/Core/Ui/Dialog", "WoltLabSuite/Core/Dom/Util"], function (require, exports, tslib_1, Ajax, Core, EventHandler, Language, LanguageInput, Dialog_1, Util_1) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.closeDialog = exports.loadDialog = exports.hasI18n = exports.setI18n = void 0;
    Ajax = tslib_1.__importStar(Ajax);
    Core = tslib_1.__importStar(Core);
    EventHandler = tslib_1.__importStar(EventHandler);
    Language = tslib_1.__importStar(Language);
    LanguageInput = tslib_1.__importStar(LanguageInput);
    Dialog_1 = tslib_1.__importDefault(Dialog_1);
    Util_1 = tslib_1.__importDefault(Util_1);
    class AcpUiBoardThreadFormOptionDialogHandler {
        constructor() {
            this.callbackDialogSave = undefined;
            this.dialogData = undefined;
            this.dialogLoaded = false;
            this.hasI18nValues = false;
        }
        /**
         * Sets an internal flag variable whether the installation has multiple languages and therefore the i18n fields need to be loaded or not.
         */
        setI18n(i18n) {
            this.hasI18nValues = i18n;
        }
        /**
         * Returns true, if the form has i18n fields.
         */
        hasI18n() {
            return this.hasI18nValues;
        }
        /**
         * Open the edit/add dialog for options.
         */
        loadDialog(saveCallback, data, title) {
            this.callbackDialogSave = saveCallback;
            if (!this.dialogLoaded) {
                Ajax.api(this);
                this.dialogData = data;
                this.dialogTitle = title;
            }
            else {
                if (data !== null && data !== undefined) {
                    this.setData(data);
                }
                Dialog_1.default.open(this);
                Dialog_1.default.setTitle(this, title);
            }
        }
        /**
         * Closes the current dialog.
         */
        closeDialog() {
            Dialog_1.default.close(this);
        }
        /**
         * Submit the current dialog and trigger the saveCallback, if there are no validation errors.
         */
        submitDialog(event) {
            event.preventDefault();
            const submitButton = this.getDialogContent().querySelector('input[type="submit"]');
            submitButton.disabled = true;
            this.resetErrors();
            if (this.validateDialog()) {
                this.saveDialog();
            }
            else {
                submitButton.disabled = false;
            }
        }
        /**
         * Validates the current dialog.
         */
        validateDialog() {
            let validationFailed = false;
            EventHandler.fire("com.woltlab.wbb.threadFormOptionDialogHandler", "validateDialog", {
                dialogData: this.getDialogData(),
                validationFailed,
            });
            if (!(this.getDialogData()["optionTitle"] !== undefined && this.getDialogData()["optionTitle"].trim() !== "")) {
                if (this.hasI18nValues) {
                    if (!LanguageInput.validate("optionTitle", false)) {
                        if (LanguageInput.validate("optionTitle", true)) {
                            this.throwError("optionTitle", Language.get("wcf.global.form.error.empty"));
                        }
                        else {
                            this.throwError("optionTitle", Language.get("wcf.global.form.error.multilingual"));
                        }
                        validationFailed = true;
                    }
                }
                else {
                    this.throwError("optionTitle", Language.get("wcf.global.form.error.empty"));
                    validationFailed = true;
                }
            }
            if (this.hasI18nValues && !LanguageInput.validate("optionDescription", true)) {
                this.throwError("optionDescription", Language.get("wcf.global.form.error.empty"));
                validationFailed = true;
            }
            return !validationFailed;
        }
        /**
         * Throw an error in the current dialog.
         */
        throwError(errorFieldName, errorMessageText) {
            const inputElement = document.getElementById(errorFieldName);
            if (!inputElement) {
                throw new Error(`Unknown errorFieldName '${errorFieldName}'`);
            }
            const dl = inputElement.closest("dl");
            const dd = inputElement.closest("dd");
            dl.classList.add("formError");
            const errorMessage = document.createElement("small");
            errorMessage.classList.add("innerError");
            errorMessage.innerHTML = errorMessageText;
            dd.appendChild(errorMessage);
        }
        /**
         * Reset the errors in the current dialog.
         */
        resetErrors() {
            this.getDialogContent()
                .querySelectorAll(".innerError")
                .forEach((element) => element.remove());
            this.getDialogContent()
                .querySelectorAll(".formError")
                .forEach((element) => element.classList.remove("formError"));
        }
        /**
         * Save the current dialog und call the saveCallback given on the dialog open function.
         */
        saveDialog() {
            if (typeof this.callbackDialogSave === "function") {
                this.callbackDialogSave(this.getDialogData(), this.getDialogContent());
            }
        }
        /**
         * Sets up the submit button in the current dialog.
         */
        setupSubmitButton() {
            const submitDiv = document.createElement("div");
            submitDiv.className = "formSubmit";
            const submitButton = document.createElement("input");
            submitButton.type = "submit";
            submitButton.value = Language.get("wcf.global.button.submit");
            submitButton.addEventListener("click", (event) => this.submitDialog(event));
            submitDiv.appendChild(submitButton);
            const dialogContent = this.getDialogContent();
            dialogContent.appendChild(submitDiv);
        }
        /**
         * Delete all dialog values in the current dialog.
         */
        cleanUpDialog() {
            const dialogContent = this.getDialogContent();
            EventHandler.fire("com.woltlab.wbb.threadFormOptionDialogHandler", "cleanUpDialog", {
                dialogContent: dialogContent,
            });
            this.getInputElement("showOrder").value = "0";
            this.getInputElement("defaultValue").value = "";
            this.getInputElement("validationPattern").value = "";
            this.getInputElement("required").checked = false;
            const optionType = document.getElementById("optionType");
            optionType.value = "text";
            this.getInputElement("selectOptions").value = "";
            Util_1.default.hide(document.getElementById("selectOptionsDL"));
            // reset i18n
            if (this.hasI18nValues) {
                const optionTitleValues = LanguageInput.getValues("optionTitle");
                const dummyObject = {};
                optionTitleValues.forEach((value, key) => {
                    dummyObject[key] = "";
                });
                LanguageInput.setValues("optionTitle", dummyObject);
                LanguageInput.setValues("optionDescription", dummyObject);
            }
            else {
                this.getInputElement("optionTitle").value = "";
                this.getInputElement("optionDescription").value = "";
            }
            this.resetErrors();
            const submitButton = this.getDialogContent().querySelector('input[type="submit"]');
            submitButton.disabled = false;
        }
        /**
         * Set the data for the current dialog.
         */
        setData(data) {
            EventHandler.fire("com.woltlab.wbb.threadFormOptionDialogHandler", "setData", {
                data: data,
            });
            Object.entries(data.simpleValues).forEach(([elementId, value]) => {
                this.getInputElement(elementId).value = value.toString();
            });
            Object.entries(data.checkboxValues).forEach(([elementId, value]) => {
                const element = this.getInputElement(elementId);
                element.checked = Core.stringToBool(value.toString());
            });
            Object.entries(data.selectOption).forEach(([elementId, value]) => {
                const element = document.getElementById(elementId);
                element.value = value;
            });
            Object.entries(data.i18nValues).forEach(([key, value]) => {
                if (this.hasI18nValues) {
                    // in case the dialog has been loaded and the i18n fields are not yet initialized
                    const tryToSetValue = () => {
                        try {
                            LanguageInput.setValues(key, value);
                        }
                        catch (e) {
                            setTimeout(tryToSetValue, 100);
                        }
                    };
                    tryToSetValue();
                }
                else {
                    // The `value` is a plain string when i18n is disabled.
                    this.getInputElement(key).value = value;
                }
            });
            Core.triggerEvent(this.getInputElement("optionType"), "change");
        }
        /**
         * Returns the current dialog content.
         */
        getDialogContent() {
            return Dialog_1.default.getDialog("threadFormOptionDialog").content;
        }
        /**
         * Returns the dialog data as object.
         */
        getDialogData() {
            const dialogData = {};
            const dialogContent = this.getDialogContent();
            const simpleValues = ["showOrder", "defaultValue", "validationPattern", "selectOptions", "optionType"];
            const checkboxValues = ["required", "isDisabled"];
            const i18nValues = ["optionTitle", "optionDescription"];
            EventHandler.fire("com.woltlab.wbb.threadFormOptionDialogHandler", "getDialogData", {
                dialogData,
                dialogContent,
                simpleValues,
                checkboxValues,
                i18nValues,
            });
            simpleValues.forEach((elementId) => {
                dialogData[elementId] = this.getInputElement(elementId).value;
            });
            checkboxValues.forEach((elementId) => {
                dialogData[elementId] = this.getInputElement(elementId).checked ? 1 : 0;
            });
            i18nValues.forEach((elementId) => {
                if (this.hasI18nValues) {
                    const i18nData = LanguageInput.getValues(elementId);
                    if (i18nData.get(0) !== undefined) {
                        dialogData[elementId] = i18nData.get(0);
                    }
                    else {
                        dialogData[`${elementId}_i18n`] = {};
                        i18nData.forEach((languageValue, key) => {
                            if (key === 0) {
                                return;
                            }
                            dialogData[`${elementId}_i18n`][key] = languageValue;
                        });
                    }
                }
                else {
                    dialogData[elementId] = this.getInputElement(elementId).value;
                }
            });
            return dialogData;
        }
        getInputElement(elementId) {
            const element = document.getElementById(elementId);
            if (element === null) {
                throw new Error(`Unable to find an element identifier by '${elementId}'.`);
            }
            return element;
        }
        _ajaxSuccess(data) {
            if (data.actionName === "getDialogAddForm") {
                this.dialogLoaded = true;
                Dialog_1.default.open(this, data.returnValues.template);
                Dialog_1.default.setTitle(this, this.dialogTitle);
                this.setupSubmitButton();
                if (this.dialogData) {
                    this.setData(this.dialogData);
                    this.dialogData = undefined;
                }
                Dialog_1.default.rebuild(this);
            }
        }
        _ajaxSetup() {
            return {
                data: {
                    className: "wbb\\data\\thread\\form\\option\\ThreadFormOptionAction",
                    actionName: "getDialogAddForm",
                },
            };
        }
        _dialogSetup() {
            return {
                id: "threadFormOptionDialog",
                options: {
                    onClose: () => this.cleanUpDialog(),
                },
                source: null,
            };
        }
    }
    let dialogHandler;
    function getDialogHandler() {
        if (!dialogHandler) {
            dialogHandler = new AcpUiBoardThreadFormOptionDialogHandler();
        }
        return dialogHandler;
    }
    /**
     * Sets an internal flag variable whether the installation has multiple languages and therefore the i18n fields need to be loaded or not.
     */
    function setI18n(i18n) {
        getDialogHandler().setI18n(i18n);
    }
    exports.setI18n = setI18n;
    /**
     * Returns true, if the form has i18n fields.
     */
    function hasI18n() {
        return getDialogHandler().hasI18n();
    }
    exports.hasI18n = hasI18n;
    /**
     * Open the edit/add dialog for options.
     */
    function loadDialog(saveCallback, data, title) {
        getDialogHandler().loadDialog(saveCallback, data, title);
    }
    exports.loadDialog = loadDialog;
    /**
     * Closes the current dialog.
     */
    function closeDialog() {
        getDialogHandler().closeDialog();
    }
    exports.closeDialog = closeDialog;
});
