<?php

namespace wbb\system\event\listener;

use wbb\data\board\BoardCache;
use wbb\data\post\PostList;
use wbb\data\thread\ThreadList;
use wcf\system\bbcode\BBCodeHandler;
use wcf\system\event\listener\AbstractHtmlInputNodeProcessorListener;
use wcf\system\html\input\node\HtmlInputNodeProcessor;
use wcf\system\Regex;
use wcf\system\request\LinkHandler;
use wcf\util\Url;

/**
 * Parses URLs of posts, threads and boards.
 *
 * @author  Tim Duesterhus, Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Event\Listener
 */
class HtmlInputNodeProcessorListener extends AbstractHtmlInputNodeProcessorListener
{
    /**
     * @inheritDoc
     */
    public function execute($eventObj, $className, $eventName, array &$parameters)
    {
        /** @var HtmlInputNodeProcessor $eventObj */

        // replace post links
        $regex = $this->getIdParameterRegexFromLink(LinkHandler::getInstance()->getLink('Thread', [
            'application' => 'wbb',
            'forceFrontend' => true,
        ]), 'postID');
        $postIDs = $this->getObjectIDs($eventObj, $regex);
        if (!empty($postIDs)) {
            $postList = new PostList();
            $postList->setObjectIDs($postIDs);
            $postList->readObjects();

            $posts = [];
            // check permissions
            foreach ($postList as $post) {
                if ($post->canRead()) {
                    $posts[$post->postID] = $post;
                }
            }

            $this->replaceLinks(
                $eventObj,
                $posts,
                BBCodeHandler::getInstance()->isAvailableBBCode('post') ? 'post' : ''
            );
        }

        // replace thread links
        $regex = $this->getRegexFromLink(LinkHandler::getInstance()->getLink('Thread', [
            'application' => 'wbb',
            'forceFrontend' => true,
        ]));
        $threadIDs = $this->getObjectIDs($eventObj, $regex);
        if (!empty($threadIDs)) {
            $threadList = new ThreadList();
            $threadList->setObjectIDs($threadIDs);
            $threadList->readObjects();

            $threads = [];
            // check permissions
            foreach ($threadList as $thread) {
                if ($thread->canRead()) {
                    $threads[$thread->threadID] = $thread;
                }
            }

            $this->replaceLinks(
                $eventObj,
                $threads,
                BBCodeHandler::getInstance()->isAvailableBBCode('thread') ? 'thread' : ''
            );
        }

        // replace board links
        $regex = $this->getRegexFromLink(LinkHandler::getInstance()->getLink('Board', [
            'application' => 'wbb',
            'forceFrontend' => true,
        ]));
        $boardIDs = $this->getObjectIDs($eventObj, $regex);

        if (!empty($boardIDs)) {
            $boards = [];
            foreach ($boardIDs as $boardID) {
                $board = BoardCache::getInstance()->getBoard($boardID);
                if ($board !== null && $board->canEnter()) {
                    $boards[$board->boardID] = $board;
                }
            }

            $this->replaceLinks($eventObj, $boards);
        }
    }

    /**
     * @inheritDoc
     */
    protected function getRegexFromLink($link, $defaultAnchor = '')
    {
        return new Regex('^(' . \preg_replace('~^https?~', 'https?', \preg_quote($link)) . '(\d+)-.*?)$');
    }

    /**
     * Builds a Regex that matches the given integer-valued parameter
     * with name `$parameter` in the given base `$link`.
     *
     *
     * @param   string  $link
     * @param   string  $parameter
     * @return  Regex
     */
    protected function getIdParameterRegexFromLink($link, $parameter)
    {
        $separator = '\?';
        if (Url::parse($link)['query']) {
            $separator = '&';
        }

        return new Regex('^(' . \preg_replace('~^https?~', 'https?', \preg_quote($link)) . '.*?' . $separator . \preg_quote($parameter) . '=(\d+)(?:[&#].*?)?)$');
    }
}
