<?php

namespace wbb\system\user\notification\event;

use wbb\system\cache\runtime\ThreadRuntimeCache;
use wbb\system\thread\status\ThreadStatusHandler;
use wbb\system\user\notification\object\PostUserNotificationObject;
use wcf\system\email\Email;
use wcf\system\user\notification\event\AbstractSharedUserNotificationEvent;
use wcf\system\user\notification\event\ITestableUserNotificationEvent;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\WCF;

/**
 * Notification event for posts.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\User\Notification\Event
 *
 * @method  PostUserNotificationObject  getUserNotificationObject()
 */
class PostUserNotificationEvent extends AbstractSharedUserNotificationEvent implements ITestableUserNotificationEvent
{
    use TTestablePostUserNotificationEvent;

    /**
     * @inheritDoc
     */
    protected $stackable = true;

    /**
     * @inheritDoc
     */
    protected function prepare()
    {
        ThreadRuntimeCache::getInstance()->cacheObjectID($this->getUserNotificationObject()->threadID);
    }

    /**
     * @inheritDoc
     */
    public function getTitle()
    {
        $count = \count($this->getAuthors());
        if ($count > 1) {
            return $this->getLanguage()->getDynamicVariable('wbb.post.notification.title.stacked', [
                'count' => $count,
                'timesTriggered' => $this->notification->timesTriggered,
            ]);
        }

        return $this->getLanguage()->get('wbb.post.notification.title');
    }

    /**
     * @inheritDoc
     */
    public function getMessage()
    {
        $authors = $this->getAuthors();
        $count = \count($authors);

        if ($count > 1) {
            if (isset($authors[0])) {
                unset($authors[0]);
            }
            $count = \count($authors);

            return $this->getLanguage()->getDynamicVariable('wbb.post.notification.message.stacked', [
                'author' => $this->author,
                'authors' => \array_values($authors),
                'count' => $count,
                'guestTimesTriggered' => $this->notification->guestTimesTriggered,
                'message' => $this->getUserNotificationObject(),
                'others' => $count - 1,
            ]);
        }

        return $this->getLanguage()->getDynamicVariable('wbb.post.notification.message', [
            'userNotificationObject' => $this->getUserNotificationObject(),
            'author' => $this->author,
        ]);
    }

    /**
     * @inheritDoc
     */
    public function getEmailMessage($notificationType = 'instant')
    {
        $this->getUserNotificationObject()->setThread(
            ThreadRuntimeCache::getInstance()->getObject($this->getUserNotificationObject()->threadID)
        );

        $messageID = '<com.woltlab.wbb.thread/' . $this->getUserNotificationObject()->getThread()->threadID . '@' . Email::getHost() . '>';

        return [
            'message-id' => 'com.woltlab.wbb.post/' . $this->getUserNotificationObject()->postID,
            'template' => 'email_notification_post',
            'application' => 'wbb',
            'in-reply-to' => [$messageID],
            'references' => [$messageID],
            'variables' => [
                'post' => $this->getUserNotificationObject(),
                'thread' => $this->getUserNotificationObject()->getThread(),
                'author' => $this->author,
            ],
        ];
    }

    /**
     * @inheritDoc
     * @since   5.0
     */
    public function getEmailTitle()
    {
        if (\count($this->getAuthors()) > 1) {
            return parent::getEmailTitle();
        }

        $this->getUserNotificationObject()->setThread(
            ThreadRuntimeCache::getInstance()->getObject($this->getUserNotificationObject()->threadID)
        );

        return $this->getLanguage()->getDynamicVariable('wbb.post.notification.mail.title', [
            'userNotificationObject' => $this->getUserNotificationObject(),
            'thread' => $this->getUserNotificationObject()->getThread(),
        ]);
    }

    /**
     * @inheritDoc
     */
    public function getEventHash()
    {
        return \sha1($this->eventID . '-' . $this->getUserNotificationObject()->threadID);
    }

    /**
     * @inheritDoc
     */
    public function getLink()
    {
        return $this->getUserNotificationObject()->getLink();
    }

    /**
     * @inheritDoc
     */
    public function checkAccess()
    {
        $this->getUserNotificationObject()->setThread(
            ThreadRuntimeCache::getInstance()->getObject($this->getUserNotificationObject()->threadID)
        );

        $returnValue = $this->getUserNotificationObject()->canRead();
        if (!$this->getUserNotificationObject()->getThread()->canRead()) {
            // remove subscription
            ThreadStatusHandler::saveSubscriptionStatus(
                [$this->getUserNotificationObject()->threadID],
                ThreadStatusHandler::SUBSCRIPTION_MODE_NORMAL,
                WCF::getUser()
            );

            // reset user storage
            UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedThreads');

            $returnValue = false;
        }

        return $returnValue;
    }
}
