/**
 * Handles toggling watched threads in the list of all watched threads.
 *
 * @author  Joshua Ruesweg
 * @copyright  2001-2021 WoltLab GmbH
 * @license  WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module  WoltLabSuite/Forum/Controller/Thread/WatchedList
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Ajax", "WoltLabSuite/Core/Core", "WoltLabSuite/Core/Language", "WoltLabSuite/Core/Ui/Confirmation"], function (require, exports, tslib_1, Ajax, Core, Language, UiConfirmation) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.setup = void 0;
    Ajax = tslib_1.__importStar(Ajax);
    Core = tslib_1.__importStar(Core);
    Language = tslib_1.__importStar(Language);
    UiConfirmation = tslib_1.__importStar(UiConfirmation);
    let _button;
    const _checkboxes = [];
    let _markAllButton;
    /**
     * Returns the ids of the selected watched threads.
     */
    function getSelectedThreadIDs() {
        return _checkboxes.filter((checkbox) => checkbox.checked).map((checkbox) => ~~checkbox.dataset.objectId);
    }
    /**
     * Handles changing a mark checkbox.
     */
    function mark(event) {
        updateActionButtonLabel();
        const checkbox = event.currentTarget;
        updateMarkAllButton();
        if (checkbox.checked) {
            checkbox.closest("ol").classList.add("jsMarked");
        }
        else {
            checkbox.closest("ol").classList.remove("jsMarked");
        }
    }
    /**
     * Handles clicking on the 'mark all' button.
     */
    function markAll() {
        const isChecked = Core.stringToBool(_markAllButton.dataset.hasUnmarkedThreads);
        _checkboxes.forEach((checkbox) => {
            checkbox.checked = isChecked;
            if (isChecked) {
                checkbox.closest("ol").classList.add("jsMarked");
            }
            else {
                checkbox.closest("ol").classList.remove("jsMarked");
            }
        });
        updateActionButtonLabel();
        updateMarkAllButton();
    }
    /**
     * Handles a click on the stop watching button.
     */
    function stopWatching() {
        const threadIDs = getSelectedThreadIDs();
        const type = threadIDs.length ? "stopWatchingMarked" : "stopWatchingAll";
        const languageItem = `wbb.thread.watchedThreads.${type}.confirmMessage`;
        UiConfirmation.show({
            confirm() {
                Ajax.apiOnce({
                    data: {
                        actionName: "stopWatching",
                        className: "wbb\\data\\thread\\ThreadAction",
                        parameters: {
                            threadIDs,
                            stopWatchingAll: getSelectedThreadIDs().length ? 0 : 1,
                        },
                    },
                    success() {
                        window.location.reload();
                    },
                });
            },
            message: Language.get(languageItem),
        });
    }
    /**
     * Updates the label of the action button.
     */
    function updateActionButtonLabel() {
        const count = getSelectedThreadIDs().length;
        if (count) {
            _button.textContent = Language.get("wbb.thread.watchedThreads.stopWatchingMarked", {
                count: count,
            });
        }
        else {
            _button.textContent = Language.get("wbb.thread.watchedThreads.stopWatchingAll");
        }
    }
    /**
     * Updates the label of mark all button.
     */
    function updateMarkAllButton() {
        const hasUnmarkedThreads = _checkboxes.some((checkbox) => !checkbox.checked);
        _markAllButton.dataset.hasUnmarkedThreads = hasUnmarkedThreads ? "true" : "false";
        if (hasUnmarkedThreads) {
            _markAllButton.closest("ol").classList.remove("jsMarked");
        }
        else {
            _markAllButton.closest("ol").classList.add("jsMarked");
        }
    }
    /**
     * Initializes the watched thread list.
     */
    function setup() {
        document.querySelectorAll(".jsWatchedThread").forEach((checkbox) => {
            checkbox.addEventListener("change", mark);
            _checkboxes.push(checkbox);
        });
        _button = document.getElementById("stopWatchingButton");
        _button.addEventListener("click", stopWatching);
        _markAllButton = document.querySelector(".jsMarkAllWatchedThreads");
        _markAllButton.addEventListener("click", markAll);
        _markAllButton.dataset.hasUnmarkedThreads = "1";
    }
    exports.setup = setup;
});
