<?php

namespace wbb\data\post;

use wbb\data\board\BoardCache;
use wbb\data\thread\Thread;
use wcf\data\attachment\GroupedAttachmentList;
use wcf\data\DatabaseObject;
use wcf\data\IMessage;
use wcf\data\IPollObject;
use wcf\data\IPopoverObject;
use wcf\data\poll\Poll;
use wcf\data\TUserContent;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\NamedUserException;
use wcf\system\flood\FloodControl;
use wcf\system\html\output\AmpHtmlOutputProcessor;
use wcf\system\html\output\HtmlOutputProcessor;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;
use wcf\util\MessageUtil;
use wcf\util\StringUtil;
use wcf\util\UserUtil;

/**
 * Represents a post.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Post
 *
 * @property-read   int     $postID         unique id of the post
 * @property-read   int     $threadID       id of the thread the post belongs to
 * @property-read   int|null    $userID         id of the user who wrote the post or null if the user does not exist anymore or if post has been created by guest
 * @property-read   string      $username       name of the user or guest who wrote the post
 * @property-read   string      $subject        subject of the post
 * @property-read   string      $message        text of the post
 * @property-read   int     $time           timestamp at which the post has been created
 * @property-read   int     $isDeleted      is 1 if the post is in trash bin, otherwise 0
 * @property-read   int     $isDisabled     is 1 if the post is disabled, otherwise 0
 * @property-read   int     $isClosed       is 1 if the post may not be edited, otherwise 0
 * @property-read   int     $isOfficial     is 1 if the post is marked official, otherwise 0
 * @property-read   int|null    $editorID       id of the last user who edited the post or null if nobody edited the post or last editor has been deleted
 * @property-read   string      $editor         name of the last user who edited the post
 * @property-read   int     $lastEditTime       last timestamp at which the post has been edited
 * @property-read   int     $editCount      number of times the post has been edited
 * @property-read   string      $editReason     reason for editing the post
 * @property-read   int     $lastVersionTime    timestamp of the last edit history entry
 * @property-read   int     $attachments        number of attachments
 * @property-read   int|null    $pollID         id of the poll attached to the poll or null if no poll is attached to the post
 * @property-read   int     $enableHtml     is 1 if HTML will rendered in the post, otherwise 0
 * @property-read   string      $ipAddress      ip address of the user who wrote the post at time of post creation or empty if no ip addresses are logged
 * @property-read   int     $cumulativeLikes    cumulative result of likes (counting +1) and dislikes (counting -1)
 * @property-read   int     $deleteTime     timestamp at which the post has been deleted
 * @property-read   int     $enableTime     timestamp at which the post will automatically be enabled or 0 if post will not or already has been automatically enabled
 * @property-read   int     $hasEmbeddedObjects number of embedded objects in post
 */
class Post extends DatabaseObject implements IMessage, IPollObject, IPopoverObject
{
    use TUserContent;

    /**
     * poll object
     * @var Poll
     */
    protected $poll;

    /**
     * thread object
     * @var Thread
     */
    protected $thread;

    /**
     * thread form option values
     * @var array
     */
    protected $optionValues;

    /**
     * @inheritDoc
     */
    public function getMessage()
    {
        return $this->message;
    }

    /**
     * @inheritDoc
     */
    public function getFormattedMessage()
    {
        $processor = new HtmlOutputProcessor();
        $processor->process($this->message, 'com.woltlab.wbb.post', $this->postID);

        return $processor->getHtml();
    }

    /**
     * Returns an excerpt of the formatted message text.
     *
     * @param int $maxLength
     * @param bool $removeLinks
     * @return string
     * @since 5.2
     */
    public function getFormattedExcerpt($maxLength = 500, $removeLinks = false)
    {
        $processor = new HtmlOutputProcessor();
        $processor->removeLinks = $removeLinks;
        $processor->process($this->message, 'com.woltlab.wbb.post', $this->postID);

        return MessageUtil::truncateFormattedMessage($processor->getHtml(), $maxLength);
    }

    /**
     * Returns a simplified version of the formatted message.
     *
     * @return  string
     */
    public function getSimplifiedFormattedMessage()
    {
        $processor = new HtmlOutputProcessor();
        $processor->setOutputType('text/simplified-html');
        $processor->process($this->message, 'com.woltlab.wbb.post', $this->postID);

        return $processor->getHtml();
    }

    /**
     * Returns the post's formatted content ready for use with Google AMP pages.
     *
     * @return      string
     */
    public function getAmpFormattedContent()
    {
        $processor = new AmpHtmlOutputProcessor();
        $processor->process($this->message, 'com.woltlab.wbb.post', $this->postID);

        return $processor->getHtml();
    }

    /**
     * Returns the post's content in plain text.
     *
     * @return      string
     * @since       5.2
     */
    public function getPlainTextMessage()
    {
        $processor = new HtmlOutputProcessor();
        $processor->setOutputType('text/plain');
        $processor->process($this->message, 'com.woltlab.wbb.post', $this->postID);

        return $processor->getHtml();
    }

    /**
     * Returns and assigns embedded attachments.
     *
     * @param   bool        $ignoreCache
     * @return  null|GroupedAttachmentList
     */
    public function getAttachments($ignoreCache = false)
    {
        if ($this->attachments || $ignoreCache) {
            $attachmentList = new GroupedAttachmentList('com.woltlab.wbb.post');
            $attachmentList->getConditionBuilder()->add('attachment.objectID IN (?)', [$this->postID]);
            $attachmentList->readObjects();

            if ($ignoreCache && !\count($attachmentList)) {
                return null;
            }

            return $attachmentList;
        }

        return null;
    }

    /**
     * Returns an excerpt of this post.
     *
     * @param   int     $maxLength
     * @return  string
     */
    public function getExcerpt($maxLength = 255)
    {
        return StringUtil::truncateHTML($this->getSimplifiedFormattedMessage(), $maxLength);
    }

    /**
     * Returns true if the active user has the permission to read this post.
     *
     * @return  bool
     */
    public function canRead()
    {
        $board = BoardCache::getInstance()->getBoard($this->getThread()->boardID);

        if (!$board->canEnter()) {
            return false;
        }

        if (
            !$board->getPermission('canReadThread')
            || ($this->isDeleted && !$board->getModeratorPermission('canReadDeletedPost'))
            || ($this->isDisabled
                && !$board->getModeratorPermission('canEnablePost')
                && (!$this->userID || $this->userID != WCF::getUser()->userID)
            )
        ) {
            return false;
        }

        if ($board->isPrivate) {
            if (!WCF::getUser()->userID) {
                return false;
            }

            if (!$board->canReadPrivateThreads() && $this->getThread()->userID != WCF::getUser()->userID) {
                return false;
            }
        }

        return true;
    }

    /**
     * Returns true if active user has the permission to delete this post.
     *
     * @return  bool
     */
    public function canDelete()
    {
        $thread = $this->getThread();
        $board = BoardCache::getInstance()->getBoard($thread->boardID);

        $isModerator = $board->getModeratorPermission('canDeletePost');
        $isAuthor = $this->userID && $this->userID == WCF::getUser()->userID;
        $canDeletePost = $isModerator || ($isAuthor && $board->getPermission('canDeleteOwnPost'));

        if (!$canDeletePost || (!$isModerator && ($board->isClosed || $thread->isClosed || $this->isClosed))) {
            return false;
        }

        return true;
    }

    /**
     * Returns true if active user has the permission to completely delete this post.
     *
     * @return  bool
     */
    public function canDeleteCompletely()
    {
        $thread = $this->getThread();
        $board = BoardCache::getInstance()->getBoard($thread->boardID);

        if (!$board->getModeratorPermission('canDeletePostCompletely')) {
            return false;
        }

        return true;
    }

    /**
     * Returns true if active user has the permission to restore this post.
     *
     * @return  bool
     */
    public function canRestore()
    {
        $thread = $this->getThread();
        $board = BoardCache::getInstance()->getBoard($thread->boardID);

        if (!$board->getModeratorPermission('canRestorePost')) {
            return false;
        }

        return true;
    }

    /**
     * Returns true if active user has the permission to enable this post.
     *
     * @return  bool
     */
    public function canEnable()
    {
        $thread = $this->getThread();
        $board = BoardCache::getInstance()->getBoard($thread->boardID);

        if (!$board->getModeratorPermission('canEnablePost')) {
            return false;
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function isVisible()
    {
        return $this->canRead();
    }

    /**
     * Returns true, if the post object is the best answer in the thread.
     *
     * @return      bool
     * @since       5.2
     */
    public function isBestAnswer()
    {
        return $this->getThread()->bestAnswerPostID == $this->postID;
    }

    /**
     * Returns true, if the post is marked official.
     *
     * @since       5.4
     */
    public function isOfficial(): bool
    {
        return (bool)$this->isOfficial;
    }

    /**
     * Returns the thread of this post.
     *
     * @return  Thread
     */
    public function getThread()
    {
        if ($this->thread === null) {
            $this->thread = new Thread($this->threadID);
        }

        return $this->thread;
    }

    /**
     * Sets thread object for this post.
     *
     * @param   Thread      $thread
     */
    public function setThread(Thread $thread)
    {
        if ($thread->threadID == $this->threadID) {
            $this->thread = $thread;
        }
    }

    /**
     * Returns true if this post is the first post in its thread.
     *
     * @return  bool
     */
    public function isFirstPost()
    {
        return $this->getThread()->firstPostID == $this->postID;
    }

    /**
     * Returns a specific thread form option value.
     *
     * @param   int     $optionID
     * @return  string
     * @since       5.2
     */
    public function getOptionValue($optionID)
    {
        if ($this->optionValues === null) {
            $this->optionValues = [];
            $sql = "SELECT  optionID, optionValue
                    FROM    wbb" . WCF_N . "_thread_form_option_value
                    WHERE   postID = ?";
            $statement = WCF::getDB()->prepareStatement($sql);
            $statement->execute([$this->postID]);

            $this->optionValues = $statement->fetchMap('optionID', 'optionValue');
        }

        if (isset($this->optionValues[$optionID])) {
            return $this->optionValues[$optionID];
        }

        return '';
    }

    /**
     * Returns true if this post has got old versions in edit history.
     *
     * @return  bool
     */
    public function hasOldVersions()
    {
        if (!MODULE_EDIT_HISTORY) {
            return false;
        }
        if (EDIT_HISTORY_EXPIRATION == 0) {
            return $this->lastVersionTime > 0;
        }

        return $this->lastVersionTime > (TIME_NOW - EDIT_HISTORY_EXPIRATION * 86400);
    }

    /**
     * Returns a list of ip addresses used by a user.
     *
     * @param   int     $userID
     * @param   string      $username
     * @param   string      $notIpAddress
     * @param   int     $limit
     * @return  mixed[][]
     */
    public static function getIpAddressByAuthor($userID, $username = '', $notIpAddress = '', $limit = 10)
    {
        $conditions = new PreparedStatementConditionBuilder();
        $conditions->add("userID = ?", [$userID]);
        if (!empty($username) && !$userID) {
            $conditions->add("username = ?", [$username]);
        }
        if (!empty($notIpAddress)) {
            $conditions->add("ipAddress <> ?", [$notIpAddress]);
        }
        $conditions->add("ipAddress <> ''");

        $sql = "SELECT      ipAddress, MAX(time) AS time
                FROM        wbb" . WCF_N . "_post
                " . $conditions . "
                GROUP BY    ipAddress
                ORDER BY    MAX(time) DESC, MAX(postID) DESC";
        $statement = WCF::getDB()->prepareStatement($sql, $limit);
        $statement->execute($conditions->getParameters());

        $ipAddresses = [];
        while ($row = $statement->fetchArray()) {
            $ipAddresses[] = $row;
        }

        return $ipAddresses;
    }

    /**
     * Returns a list of users which have used the given ip address.
     *
     * @param   string      $ipAddress
     * @param   int     $notUserID
     * @param   string      $notUsername
     * @param   int     $limit
     * @return  mixed[][]
     */
    public static function getAuthorByIpAddress($ipAddress, $notUserID = 0, $notUsername = '', $limit = 10)
    {
        $conditions = new PreparedStatementConditionBuilder();
        $conditions->add("ipAddress = ?", [$ipAddress]);
        if ($notUserID) {
            $conditions->add("(userID <> ? OR userID IS NULL)", [$notUserID]);
        }
        if (!empty($notUsername)) {
            $conditions->add("username <> ?", [$notUsername]);
        }

        $sql = "SELECT      username, userID, MAX(time) AS time
                FROM        wbb" . WCF_N . "_post
                " . $conditions . "
                GROUP BY    username, userID
                ORDER BY    MAX(time) DESC, MAX(postID) DESC";
        $statement = WCF::getDB()->prepareStatement($sql, $limit);
        $statement->execute($conditions->getParameters());

        $users = [];
        while ($row = $statement->fetchArray()) {
            $users[] = $row;
        }

        return $users;
    }

    /**
     * Returns post's ip address, converts into IPv4 if applicable.
     *
     * @return  string
     */
    public function getIpAddress()
    {
        if ($this->ipAddress) {
            return UserUtil::convertIPv6To4($this->ipAddress);
        }

        return '';
    }

    /**
     * @inheritDoc
     */
    public function getLink()
    {
        return LinkHandler::getInstance()->getLink('Thread', [
            'application' => 'wbb',
            'object' => $this->getThread(),
            'postID' => $this->postID,
            'forceFrontend' => true,
        ], '#post' . $this->postID);
    }

    /**
     * @inheritDoc
     */
    public function getTitle()
    {
        if ($this->subject) {
            return $this->subject;
        }

        return 'RE: ' . $this->getThread()->topic;
    }

    /**
     * @inheritDoc
     */
    public function __toString()
    {
        return $this->getFormattedMessage();
    }

    /**
     * Returns the poll object for this post.
     *
     * @return  Poll
     */
    public function getPoll()
    {
        if ($this->pollID && $this->poll === null) {
            $this->poll = new Poll($this->pollID);
            $this->poll->setRelatedObject($this);
        }

        return $this->poll;
    }

    /**
     * Sets the poll object for this post.
     *
     * @param   Poll    $poll
     */
    public function setPoll(Poll $poll)
    {
        $this->poll = $poll;

        $this->poll->setRelatedObject($this);
    }

    /**
     * @inheritDoc
     */
    public function canVote()
    {
        return $this->getThread()->getBoard()->getPermission('canVotePoll');
    }

    /**
     * Returns a version of this message optimized for use in emails.
     *
     * @param   string  $mimeType   Either 'text/plain' or 'text/html'
     * @return  string
     */
    public function getMailText($mimeType = 'text/plain')
    {
        switch ($mimeType) {
            case 'text/plain':
                return $this->getPlainTextMessage();
            case 'text/html':
                return $this->getSimplifiedFormattedMessage();
        }

        throw new \LogicException('Unreachable');
    }

    /**
     * @inheritDoc
     */
    public function getPopoverLinkClass()
    {
        return 'wbbPostLink';
    }

    /**
     * @since 5.5
     */
    public function canReact(): bool
    {
        return MODULE_LIKE
            && WCF::getUser()->userID
            && WCF::getSession()->getPermission('user.like.canLike')
            && $this->getThread()->getBoard()->getPermission('canLikePost')
            && $this->userID != WCF::getUser()->userID;
    }

    /**
     * Enforces the flood control.
     */
    public static function enforceFloodControl()
    {
        $floodControlTime = WCF::getSession()->getPermission('user.board.floodControlTime');
        if (!$floodControlTime) {
            return;
        }

        $lastTime = FloodControl::getInstance()->getLastTime('com.woltlab.wbb.post');
        if ($lastTime !== null && $lastTime > TIME_NOW - $floodControlTime) {
            throw new NamedUserException(WCF::getLanguage()->getDynamicVariable(
                'wbb.post.error.floodControl',
                ['lastPostTime' => $lastTime]
            ));
        }
    }
}
