<?php

namespace wbb\data\post;

use wbb\data\modification\log\ViewablePostModificationLog;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\user\UserProfile;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\WCF;

/**
 * Represents a viewable post.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Post
 *
 * @method  Post    getDecoratedObject()
 * @mixin   Post
 */
class ViewablePost extends DatabaseObjectDecorator
{
    /**
     * @inheritDoc
     */
    protected static $baseClass = Post::class;

    /**
     * modification log entry
     * @var ViewablePostModificationLog
     */
    protected $logEntry;

    /**
     * user profile object
     * @var UserProfile
     */
    protected $userProfile;

    /**
     * Returns the profile object of the user who created the post.
     *
     * @return  UserProfile
     */
    public function getUserProfile()
    {
        if ($this->userProfile === null) {
            if ($this->userID) {
                $this->userProfile = UserProfileRuntimeCache::getInstance()->getObject($this->userID);
            } else {
                $this->userProfile = UserProfile::getGuestUserProfile($this->username);
            }
        }

        return $this->userProfile;
    }

    /**
     * Sets modification log entry.
     *
     * @param   ViewablePostModificationLog     $logEntry
     */
    public function setLogEntry(ViewablePostModificationLog $logEntry)
    {
        if ($logEntry->objectID == $this->postID) {
            $this->logEntry = $logEntry;
        }
    }

    /**
     * Returns modification log entry.
     *
     * @return  ViewablePostModificationLog
     */
    public function getLogEntry()
    {
        return $this->logEntry;
    }

    /**
     * Returns delete note.
     *
     * @return  string
     */
    public function getDeleteNote()
    {
        if ($this->logEntry === null) {
            return WCF::getLanguage()->get('wbb.post.deleted.noData');
        }

        return WCF::getLanguage()->getDynamicVariable('wbb.post.deleted', ['post' => $this]);
    }

    /**
     * Calculates the post number relative to the related thread.
     *
     * @return      int         computed post number
     */
    public function getPostNumber()
    {
        if ($this->getDecoratedObject()->getThread()->firstPostID === $this->getDecoratedObject()->postID) {
            // avoid running a query if we know that this is the very first post
            return 1;
        }

        $sql = "SELECT  COUNT(*) AS count
                FROM    wbb" . WCF_N . "_post
                WHERE   threadID = ?
                    AND postID <> ?
                    AND time <= ?";
        $statement = WCF::getDB()->prepareStatement($sql);
        $statement->execute([
            $this->getDecoratedObject()->threadID,
            $this->getDecoratedObject()->postID,
            $this->getDecoratedObject()->time,
        ]);
        $count = $statement->fetchSingleColumn();

        return $count + 1;
    }
}
