<?php

namespace wbb\system\user\notification\event;

use wbb\system\cache\runtime\PostRuntimeCache;
use wbb\system\user\notification\object\PostModerationUserNotificationObject;
use wcf\data\user\UserProfile;
use wcf\system\user\notification\event\AbstractSharedUserNotificationEvent;
use wcf\system\user\notification\event\ITestableUserNotificationEvent;
use wcf\system\user\notification\event\TTestableUserNotificationEvent;

/**
 * User notification event for moderating posts.
 *
 * @author  Joshua Ruesweg
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\User\Notification\Event
 * @since   5.1
 *
 * @method  PostModerationUserNotificationObject    getUserNotificationObject()
 */
class ModeratePostUserNotificationEvent extends AbstractSharedUserNotificationEvent implements
    ITestableUserNotificationEvent
{
    use TTestablePostRelatedUserNotificationEvent;
    use TTestableUserNotificationEvent;

    /**
     * @inheritDoc
     */
    protected function prepare()
    {
        PostRuntimeCache::getInstance()->cacheObjectID($this->getUserNotificationObject()->objectID);
    }

    /**
     * @inheritDoc
     */
    public function getTitle()
    {
        return $this->getLanguage()->getDynamicVariable(
            'wbb.post.notification.moderate.' . $this->getUserNotificationObject()->action . '.notification.title'
        );
    }

    /**
     * @inheritDoc
     */
    public function getMessage()
    {
        return $this->getLanguage()->getDynamicVariable(
            'wbb.post.notification.moderate.' . $this->getUserNotificationObject()->action . '.notification.message',
            [
                'author' => $this->author,
                'post' => $this->getUserNotificationObject()->getPost(),
                'userNotificationObject' => $this->getUserNotificationObject(),
            ]
        );
    }

    /**
     * @inheritDoc
     */
    public function getLink()
    {
        return $this->getUserNotificationObject()->getURL();
    }

    /**
     * @inheritDoc
     */
    public function getEventHash()
    {
        return \sha1($this->eventID . '-' . $this->getUserNotificationObject()->getObjectID());
    }

    /**
     * @inheritDoc
     */
    public function supportsEmailNotification()
    {
        return false;
    }

    /**
     * @inheritDoc
     */
    public function checkAccess()
    {
        if (!WBB_THREAD_ENABLE_MODERATION_NOTIFICATION) {
            return false;
        }

        if (!$this->getUserNotificationObject()->getPost()) {
            return false;
        }

        if (
            !$this->getUserNotificationObject()->getPost()->canRead()
            && !(
                $this->getUserNotificationObject()->getPost()->getThread()->canRead()
                && $this->getUserNotificationObject()->action == 'trash'
            )
        ) {
            return false;
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public static function canBeTriggeredByGuests()
    {
        return true;
    }

    /**
     * @inheritDoc
     */
    public static function getTestObjects(UserProfile $recipient, UserProfile $author)
    {
        return [
            new PostModerationUserNotificationObject(
                self::createTestPostModificationLog($recipient, $author, 'close')
            ),
            new PostModerationUserNotificationObject(
                self::createTestPostModificationLog($recipient, $author, 'enable')
            ),
            new PostModerationUserNotificationObject(
                self::createTestPostModificationLog($recipient, $author, 'move')
            ),
            new PostModerationUserNotificationObject(
                self::createTestPostModificationLog($recipient, $author, 'trash', [
                    'reason' => 'Test Reason',
                ])
            ),
        ];
    }
}
