<?php

namespace wbb\system\condition\thread;

use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wbb\page\IThreadPage;
use wbb\system\board\status\BoardStatusHandler;
use wbb\system\thread\status\ThreadStatusHandler;
use wcf\data\condition\Condition;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\DatabaseObjectList;
use wcf\system\condition\AbstractSelectCondition;
use wcf\system\condition\IContentCondition;
use wcf\system\condition\IObjectCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\exception\InvalidObjectArgument;
use wcf\system\request\RequestHandler;
use wcf\system\WCF;

/**
 * Condition implementation for the subscription state of a thread.
 *
 * @author  Joshua Ruesweg, Matthias Schmidt
 * @copyright   2001-2022 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Condition\Thread
 * @since   5.2
 */
class ThreadSubscriptionCondition extends AbstractSelectCondition implements
    IContentCondition,
    IObjectCondition,
    IObjectListCondition
{
    /**
     * @inheritDoc
     */
    protected $fieldName = 'wbbThreadSubscription';

    /**
     * @inheritDoc
     */
    protected $label = 'wbb.thread.condition.subscription';

    /**
     * value of the "subscribed threads" option
     * @var int
     */
    const THREAD_SUBSCRIPTION = 1;

    /**
     * value of the "subscribed boards" option
     * @var int
     */
    const BOARD_SUBSCRIPTION = 2;

    /**
     * value of the "subscribed threads and subscribed boards" option
     * @var int
     */
    const BOARD_OR_THREAD_SUBSCRIPTION = 3;

    /**
     * @inheritDoc
     */
    public function addObjectListCondition(DatabaseObjectList $objectList, array $conditionData)
    {
        if (!($objectList instanceof ThreadList)) {
            throw new InvalidObjectArgument($objectList, ThreadList::class, 'Object list');
        }

        // only registered users can subscribe to threads or boards
        if (!WCF::getUser()->userID) {
            $objectList->getConditionBuilder()->add('1 = 0');
        } else {
            switch ($conditionData[$this->fieldName]) {
                case self::BOARD_SUBSCRIPTION:
                    $objectList->getConditionBuilder()->add(
                        "thread.boardID IN (
                            SELECT  boardID
                            FROM    wbb" . \WCF_N . "_board_user_status
                            WHERE   status IN (?)
                                AND userID = ?
                        )",
                        [
                            [
                                BoardStatusHandler::SUBSCRIPTION_MODE_THREADSONLY,
                                BoardStatusHandler::SUBSCRIPTION_MODE_THREADSANDPOSTS,
                            ],
                            WCF::getUser()->userID,
                        ]
                    );
                    break;

                case self::BOARD_OR_THREAD_SUBSCRIPTION:
                    $objectList->getConditionBuilder()->add(
                        "(thread.boardID IN (
                            SELECT  boardID
                            FROM    wbb" . \WCF_N . "_board_user_status
                            WHERE   status IN (?)
                                AND userID = ?
                        )) OR (thread.threadID IN (
                            SELECT  threadID
                            FROM    wbb" . \WCF_N . "_thread_user_status
                            WHERE   status = ?
                                AND userID = ?
                        ))",
                        [
                            [
                                BoardStatusHandler::SUBSCRIPTION_MODE_THREADSONLY,
                                BoardStatusHandler::SUBSCRIPTION_MODE_THREADSANDPOSTS,
                            ],
                            WCF::getUser()->userID,
                            ThreadStatusHandler::SUBSCRIPTION_MODE_WATCHING,
                            WCF::getUser()->userID,
                        ]
                    );
                    break;

                case self::THREAD_SUBSCRIPTION:
                    $objectList->getConditionBuilder()->add(
                        "thread.threadID IN (
                            SELECT  threadID
                            FROM    wbb" . \WCF_N . "_thread_user_status
                            WHERE   status = ?
                                AND userID = ?
                        )",
                        [
                            ThreadStatusHandler::SUBSCRIPTION_MODE_WATCHING,
                            WCF::getUser()->userID,
                        ]
                    );
                    break;
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function checkObject(DatabaseObject $object, array $conditionData)
    {
        if (
            !($object instanceof Thread)
            && (!($object instanceof DatabaseObjectDecorator) || !($object->getDecoratedObject() instanceof Thread))
        ) {
            throw new InvalidObjectArgument($object, Thread::class);
        }

        switch ($conditionData[$this->fieldName]) {
            case self::BOARD_SUBSCRIPTION:
                return $object->getBoard()->isSubscribed() || $object->getBoard()->isSubscribedForThreadsAndPosts();

            case self::BOARD_OR_THREAD_SUBSCRIPTION:
                return $object->isSubscribed() || $object->getBoard()->isSubscribed() || $object->getBoard()->isSubscribedForThreadsAndPosts();

            case self::THREAD_SUBSCRIPTION:
                return $object->isSubscribed();
        }

        throw new \LogicException("Unreachable");
    }

    /**
     * @inheritDoc
     */
    public function showContent(Condition $condition)
    {
        $requestObject = RequestHandler::getInstance()->getActiveRequest()->getRequestObject();
        if (!($requestObject instanceof IThreadPage)) {
            return false;
        }

        return $this->checkObject($requestObject->getThread(), $condition->conditionData);
    }

    /**
     * @inheritDoc
     */
    protected function getOptions()
    {
        return [
            self::NO_SELECTION_VALUE => 'wcf.global.noSelection',
            self::THREAD_SUBSCRIPTION => 'wbb.thread.condition.subscription.threadSubscription',
            self::BOARD_SUBSCRIPTION => 'wbb.thread.condition.subscription.boardSubscription',
            self::BOARD_OR_THREAD_SUBSCRIPTION => 'wbb.thread.condition.subscription.boardOrThreadSubscription',
        ];
    }
}
