<?php

namespace wbb\acp\form;

use wbb\data\board\BoardList;
use wbb\data\board\RealtimeBoardNodeList;
use wbb\data\thread\form\option\ThreadFormOption;
use wbb\data\thread\form\option\ThreadFormOptionList;
use wbb\data\thread\form\ThreadFormAction;
use wbb\data\thread\form\ThreadFormEditor;
use wcf\acp\form\AbstractAcpForm;
use wcf\system\exception\UserInputException;
use wcf\system\language\I18nValue;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;
use wcf\util\ArrayUtil;
use wcf\util\StringUtil;

/**
 * Represents the thread form add form.
 *
 * @author  Joshua Ruesweg
 * @copyright   2001-2020 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Acp\Form
 * @since   5.3
 */
class ThreadFormAddForm extends AbstractAcpForm
{
    /**
     * @inheritDoc
     */
    public $activeMenuItem = 'wbb.acp.menu.link.threadForm.add';

    /**
     * @inheritDoc
     */
    public $neededPermissions = ['admin.board.canAddBoard'];

    /**
     * @var string
     */
    public $threadFormTmpHash;

    /**
     * @var string
     */
    public $title = '';

    /**
     * @var ThreadFormOption[]
     */
    public $threadFormOptions = [];

    /**
     * @var RealtimeBoardNodeList
     */
    public $boardList;

    /**
     * Array of boardIDs were the form should be active.
     *
     * @var int[]
     */
    public $boardIDs = [];

    /**
     * @inheritDoc
     */
    public function readParameters()
    {
        parent::readParameters();

        $titleI18n = new I18nValue('title');
        $titleI18n->setLanguageItem('wbb.board.threadForm.title', 'wbb.board', 'com.woltlab.wbb');
        $this->registerI18nValue($titleI18n);

        if (empty($this->threadFormTmpHash)) {
            $this->threadFormTmpHash = StringUtil::getRandomID();
        }
    }

    /**
     * @inheritDoc
     */
    public function readFormParameters()
    {
        parent::readFormParameters();

        if (isset($_POST['threadFormTmpHash'])) {
            $this->threadFormTmpHash = $_POST['threadFormTmpHash'];

            $this->readThreadFormOptions();
        }

        if (isset($_POST['boardIDs']) && \is_array($_POST['boardIDs'])) {
            $this->boardIDs = ArrayUtil::toIntegerArray($_POST['boardIDs']);
        }
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        parent::readData();

        $this->boardList = new RealtimeBoardNodeList();
        $this->boardList->readNodeTree();
    }

    /**
     * Reads the created thread form options for the current form.
     */
    protected function readThreadFormOptions()
    {
        $threadFormOptionList = new ThreadFormOptionList();
        $threadFormOptionList->getConditionBuilder()->add('tmpHash = ?', [$this->threadFormTmpHash]);
        $threadFormOptionList->sqlOrderBy = 'thread_form_option.showOrder ASC';
        $threadFormOptionList->readObjects();

        $this->threadFormOptions = \array_merge($this->threadFormOptions, $threadFormOptionList->getObjects());
    }

    /**
     * @inheritDoc
     */
    public function validate()
    {
        parent::validate();

        if (empty($this->threadFormOptions)) {
            throw new UserInputException('threadFormFields');
        }

        $this->validateBoardIDs();
    }

    /**
     * Validate the given boardIDs.
     */
    protected function validateBoardIDs()
    {
        $boardList = new BoardList();
        $boardList->readObjects();

        foreach ($this->boardIDs as $boardID) {
            $board = $boardList->search($boardID);

            if (
                !$board
                || !$board->isBoard()
                || ($board->formID && !($this instanceof ThreadFormEditForm && $board->formID == $this->formID))
            ) {
                throw new UserInputException('boardIDs', 'noValidSelection');
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function save()
    {
        parent::save();

        $this->objectAction = new ThreadFormAction([], 'create', [
            'data' => [
                'title' => $this->title,
            ],
            'threadFormTmpHash' => $this->threadFormTmpHash,
            'boardIDs' => $this->boardIDs,
        ]);
        $this->objectAction->executeAction();

        $this->saveI18n($this->objectAction->getReturnValues()['returnValues'], ThreadFormEditor::class);

        $this->reset();

        WCF::getTPL()->assign([
            'objectEditLink' => LinkHandler::getInstance()->getControllerLink(
                ThreadFormEditForm::class,
                ['id' => $this->objectAction->getReturnValues()['returnValues']->formID]
            ),
        ]);
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        WCF::getTPL()->assign([
            'threadFormOptions' => $this->threadFormOptions,
            'threadFormTmpHash' => $this->threadFormTmpHash,
            'boardList' => $this->boardList->getNodeList(),
            'boardIDs' => $this->boardIDs,
        ]);
    }

    /**
     * @inheritDoc
     */
    public function reset()
    {
        parent::reset();

        $this->boardIDs = [];
        $this->threadFormTmpHash = StringUtil::getRandomID();
        $this->threadFormOptions = [];
    }
}
