<?php

namespace wbb\data\modification\log;

use wbb\data\thread\Thread;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\modification\log\IViewableModificationLog;
use wcf\data\modification\log\ModificationLog;
use wcf\data\user\UserProfile;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\WCF;

/**
 * Provides a viewable thread modification log.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Modification\Log
 *
 * @method  ModificationLog     getDecoratedObject()
 * @mixin   ModificationLog
 */
class ViewableThreadModificationLog extends DatabaseObjectDecorator implements IViewableModificationLog
{
    /**
     * A list of actions in which the author should not be hidden in the inline thread modification log, if
     * the author of the action is the thread author.
     * @var string[]
     * @since 5.4
     */
    public static $threadAuthorActions = ['changeTopic', 'markAsBestAnswer'];

    /**
     * @inheritDoc
     */
    protected static $baseClass = ModificationLog::class;

    /**
     * thread this modification log belongs to
     * @var Thread
     */
    protected $thread;

    /**
     * @var UserProfile
     */
    protected $userProfile;

    /**
     * Returns readable representation of current log entry.
     *
     * @return  string
     */
    public function __toString()
    {
        return WCF::getLanguage()->getDynamicVariable(
            'wbb.thread.log.thread.' . $this->action,
            ['additionalData' => $this->additionalData]
        );
    }

    /**
     * Returns the user profile object.
     *
     * @return  UserProfile
     */
    public function getUserProfile()
    {
        if ($this->userProfile === null) {
            if ($this->userID) {
                $this->userProfile = UserProfileRuntimeCache::getInstance()->getObject($this->userID);
            } else {
                $this->userProfile = UserProfile::getGuestUserProfile($this->username);
            }
        }

        return $this->userProfile;
    }

    /**
     * Returns a user guest user profile for anonymous moderator notes in threads.
     *
     * @return      UserProfile
     * @since       5.1
     */
    public function getSystemProfile()
    {
        return UserProfile::getGuestUserProfile(WCF::getLanguage()->get('wcf.user.rank.moderator'));
    }

    /**
     * Returns `false`, if the author is hidden in the inline thread modification log.
     *
     * @since   5.4
     * @throws  \BadMethodCallException If the method is called but no thread is set.
     */
    public function showAuthor(): bool
    {
        if ($this->thread === null) {
            throw new \BadMethodCallException("The method can only be called, if the thread is set.");
        }

        if (WBB_THREAD_MODERATION_NOTIFICATION_SHOW_MODERATOR) {
            return true;
        }

        if ($this->userID == $this->thread->userID && \in_array($this->action, self::$threadAuthorActions)) {
            return true;
        }

        return false;
    }

    /**
     * Sets the thread this modification log belongs to.
     *
     * @param   Thread  $thread
     */
    public function setThread(Thread $thread)
    {
        if ($thread->threadID === $this->objectID) {
            $this->thread = $thread;
        }
    }

    /**
     * @inheritDoc
     */
    public function getAffectedObject()
    {
        return $this->thread;
    }
}
