<?php

namespace wbb\data\thread\form\option;

use http\Exception\InvalidArgumentException;
use wbb\data\board\BoardCache;
use wbb\system\cache\builder\ThreadFormOptionCacheBuilder;
use wcf\system\SingletonFactory;

/**
 * Manages the thread form option cache.
 *
 * @author  Joshua Ruesweg
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Thread\Form\Option
 * @since       5.2
 */
class ThreadFormCache extends SingletonFactory
{
    /**
     * cached thread form options
     * @var ThreadFormOption[]
     */
    protected $threadFormOptions = [];

    /**
     * cached thread form options by board
     * @var ThreadFormOption[][]
     * @deprecated  since 5.3
     */
    protected $threadFormOptionsByBoard = [];

    /**
     * cached thread form options by form
     * @var ThreadFormOption[][]
     * @since 5.3
     */
    protected $threadFormOptionsByForm = [];

    /**
     * @inheritDoc
     */
    protected function init()
    {
        $this->threadFormOptions = ThreadFormOptionCacheBuilder::getInstance()
            ->getData([], 'threadFormOptions');
        $this->threadFormOptionsByForm = ThreadFormOptionCacheBuilder::getInstance()
            ->getData([], 'threadFormOptionsByForm');
    }

    /**
     * Returns a specific thread form option by id.
     *
     * @param       $optionID
     * @return      null|ThreadFormOption
     */
    public function getThreadFormOption($optionID)
    {
        return $this->threadFormOptions[$optionID] ?? null;
    }

    /**
     * Returns thread form options by board id.
     *
     * @param       int         $boardID
     * @return      ThreadFormOption[]
     * @throws \InvalidArgumentException if the given boardID is invalid
     */
    public function getThreadFormOptions($boardID)
    {
        $board = BoardCache::getInstance()->getBoard($boardID);

        if ($board === null) {
            throw new InvalidArgumentException("The given boardID '" . $boardID . "' is invalid.");
        }

        return $this->getThreadFormOptionsByFormID($board->formID);
    }

    /**
     * Returns thread form options by form id.
     *
     * @param       int         $formID
     * @return      ThreadFormOption[]
     * @since       5.3
     */
    public function getThreadFormOptionsByFormID($formID)
    {
        return $this->threadFormOptionsByForm[$formID] ?? [];
    }

    /**
     * Resets the thread form option cache.
     */
    public function resetCache()
    {
        ThreadFormOptionCacheBuilder::getInstance()->reset();
    }
}
