<?php

namespace wbb\system\condition\thread;

use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wbb\page\IThreadPage;
use wcf\data\condition\Condition;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\DatabaseObjectList;
use wcf\system\condition\AbstractIntegerCondition;
use wcf\system\condition\IContentCondition;
use wcf\system\condition\IObjectCondition;
use wcf\system\condition\IObjectListCondition;
use wcf\system\exception\InvalidObjectArgument;
use wcf\system\exception\UserInputException;
use wcf\system\request\RequestHandler;

/**
 * Condition implementation for the age (given in days) of a thread.
 *
 * @author  Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Condition\Thread
 * @since   5.2
 */
class ThreadAgeCondition extends AbstractIntegerCondition implements
    IContentCondition,
    IObjectCondition,
    IObjectListCondition
{
    /**
     * @inheritDoc
     */
    protected $description = 'wbb.thread.condition.age.description';

    /**
     * @inheritDoc
     */
    protected $fieldName = 'wbbThreadAge';

    /**
     * @inheritDoc
     */
    protected $label = 'wbb.thread.condition.age';

    /**
     * @inheritDoc
     */
    protected $identifier = 'wbbThreadAge';

    /**
     * @inheritDoc
     */
    public function addObjectListCondition(DatabaseObjectList $objectList, array $conditionData)
    {
        if (!($objectList instanceof ThreadList)) {
            throw new InvalidObjectArgument($objectList, ThreadList::class, 'Object list');
        }

        if (isset($conditionData['greaterThan'])) {
            $objectList->getConditionBuilder()->add(
                'thread.time < ?',
                [TIME_NOW - $conditionData['greaterThan'] * 86400]
            );
        }
        if (isset($conditionData['lessThan'])) {
            $objectList->getConditionBuilder()->add(
                'thread.time > ?',
                [TIME_NOW - $conditionData['lessThan'] * 86400]
            );
        }
    }

    /**
     * @inheritDoc
     */
    public function checkObject(DatabaseObject $object, array $conditionData)
    {
        if (
            !($object instanceof Thread)
            && (!($object instanceof DatabaseObjectDecorator) || !($object->getDecoratedObject() instanceof Thread))
        ) {
            throw new InvalidObjectArgument($object, Thread::class);
        }

        if (!empty($conditionData['greaterThan']) && $object->time > TIME_NOW - $conditionData['greaterThan'] * 86400) {
            return false;
        }

        if (!empty($conditionData['lessThan']) && $object->time < TIME_NOW - $conditionData['lessThan'] * 86400) {
            return false;
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function showContent(Condition $condition)
    {
        $requestObject = RequestHandler::getInstance()->getActiveRequest()->getRequestObject();
        if (!($requestObject instanceof IThreadPage)) {
            return false;
        }

        return $this->checkObject($requestObject->getThread(), $condition->conditionData);
    }

    /**
     * @inheritDoc
     */
    protected function validateConflictingValues()
    {
        if ($this->lessThan !== null && $this->greaterThan !== null && $this->greaterThan >= $this->lessThan) {
            $this->errorMessage = 'wcf.condition.greaterThan.error.lessThan';

            throw new UserInputException('greaterThan', 'lessThan');
        }
    }
}
