<?php

namespace wbb\system\page\handler;

use wbb\data\board\Board;
use wbb\system\thread\status\ThreadStatusHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\page\handler\AbstractMenuPageHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\WCF;

/**
 * Page handler for watched thread list.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Page\Handler
 * @since   5.0
 */
class WatchedThreadListPageHandler extends AbstractMenuPageHandler
{
    use TThreadStatusPageHandler;

    /**
     * number of unread watched threads.
     * @var int
     */
    protected $notifications;

    /**
     * @inheritDoc
     */
    public function isVisible($objectID = null)
    {
        return $this->hasStatusThreads(ThreadStatusHandler::SUBSCRIPTION_MODE_WATCHING, 'wbbWatchedThreads');
    }

    /**
     * @inheritDoc
     */
    public function getOutstandingItemCount($objectID = null)
    {
        if ($this->notifications === null) {
            $this->notifications = 0;

            if (WCF::getUser()->userID) {
                $data = UserStorageHandler::getInstance()->getField('wbbUnreadWatchedThreads');

                // cache does not exist or is outdated
                if ($data === null) {
                    $boardIDs = Board::filterBoardIDs(Board::getAccessibleBoardIDs());
                    $privateBoardIDs = Board::filterBoardIDs(Board::getPrivateBoardIDs());
                    if (!empty($boardIDs) || !empty($privateBoardIDs)) {
                        $conditionBuilder = new PreparedStatementConditionBuilder();

                        if (empty($privateBoardIDs)) {
                            $conditionBuilder->add("thread.boardID IN (?)", [$boardIDs]);
                        } elseif (empty($boardIDs)) {
                            $conditionBuilder->add(
                                "(thread.boardID IN (?) AND thread.userID = ?)",
                                [$privateBoardIDs, WCF::getUser()->userID]
                            );
                        } else {
                            $conditionBuilder->add(
                                "(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))",
                                [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]
                            );
                        }

                        $conditionBuilder->add(
                            'thread_user_status.status = ?',
                            [ThreadStatusHandler::SUBSCRIPTION_MODE_WATCHING]
                        );
                        $conditionBuilder->add(
                            'thread_user_status.userID = ?',
                            [WCF::getUser()->userID]
                        );
                        $conditionBuilder->add(
                            'thread.lastPostTime > ?',
                            [VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread')]
                        );
                        $conditionBuilder->add('thread.isDeleted = 0 AND thread.isDisabled = 0');
                        $conditionBuilder->add('(thread.lastPostTime > tracked_thread_visit.visitTime OR tracked_thread_visit.visitTime IS NULL)');
                        $conditionBuilder->add('(thread.lastPostTime > tracked_board_visit.visitTime OR tracked_board_visit.visitTime IS NULL)');

                        $sql = "SELECT      COUNT(*) AS count
                                FROM        wbb1_thread_user_status thread_user_status
                                LEFT JOIN   wbb1_thread thread
                                ON          thread.threadID = thread_user_status.threadID
                                LEFT JOIN   wcf1_tracked_visit tracked_thread_visit
                                ON          tracked_thread_visit.objectTypeID = ?
                                        AND tracked_thread_visit.objectID = thread.threadID
                                        AND tracked_thread_visit.userID = ?
                                LEFT JOIN   wcf1_tracked_visit tracked_board_visit
                                ON          tracked_board_visit.objectTypeID = ?
                                        AND tracked_board_visit.objectID = thread.boardID
                                        AND tracked_board_visit.userID = ?
                                " . $conditionBuilder;
                        $statement = WCF::getDB()->prepare($sql);
                        $statement->execute(\array_merge(
                            [
                                VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.thread'),
                                WCF::getUser()->userID,
                                VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.board'),
                                WCF::getUser()->userID,
                            ],
                            $conditionBuilder->getParameters()
                        ));
                        $row = $statement->fetchArray();
                        $this->notifications = $row['count'];
                    }

                    // update storage data
                    UserStorageHandler::getInstance()->update(
                        WCF::getUser()->userID,
                        'wbbUnreadWatchedThreads',
                        $this->notifications
                    );
                } else {
                    $this->notifications = $data;
                }
            }
        }

        return $this->notifications;
    }
}
