<?php

namespace wbb\system\search;

use wbb\data\board\BoardCache;
use wbb\data\board\SearchableBoardNodeList;
use wbb\data\post\SearchResultPostList;
use wbb\data\thread\SearchResultThreadList;
use wbb\data\thread\Thread;
use wbb\system\thread\status\ThreadStatusHandler;
use wcf\data\search\ISearchResultObject;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\language\LanguageFactory;
use wcf\system\search\AbstractSearchProvider;
use wcf\system\WCF;

/**
 * An implementation of ISearchProvider for searching in forum posts.
 *
 * @author  Marcel Werk
 * @copyright   2001-2021 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Search
 */
final class PostSearch extends AbstractSearchProvider
{
    /**
     * @var array
     */
    private $messageCache = [];

    /**
     * @var int
     */
    private $boardID = 0;

    /**
     * @var bool
     */
    private $findAttachments = 0;

    /**
     * @var bool
     */
    private $findPolls = 0;

    /**
     * @var bool
     */
    private $findFirstPosts = 0;

    /**
     * @var bool
     */
    private $findOfficialPosts = 0;

    /**
     * @var bool
     */
    private $findThreadsWithBestAnswer = 0;

    /**
     * shows results as threads
     * @var bool
     */
    private $findThreads = 0;

    /**
     * @var int[]
     */
    private $searchableBoardIDs = [];

    /**
     * @var int
     */
    private $threadID = 0;

    /**
     * @inheritDoc
     */
    public function cacheObjects(array $objectIDs, ?array $additionalData = null): void
    {
        if ($additionalData !== null && !empty($additionalData['findThreads'])) {
            $this->findThreads = 1;
            $threadList = new SearchResultThreadList();
            $threadList->setObjectIDs($objectIDs);
            $threadList->readObjects();
            foreach ($threadList->getObjects() as $thread) {
                $this->messageCache[$thread->threadID] = $thread;
            }
        } else {
            $postList = new SearchResultPostList();
            $postList->setObjectIDs($objectIDs);
            $postList->readObjects();
            foreach ($postList->getObjects() as $post) {
                $this->messageCache[$post->postID] = $post;
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function getObject(int $objectID): ?ISearchResultObject
    {
        return $this->messageCache[$objectID] ?? null;
    }

    /**
     * @inheritDoc
     */
    public function getFormTemplateName(): string
    {
        return 'searchPost';
    }

    /**
     * @inheritDoc
     */
    public function getJoins(): string
    {
        $join = "
            LEFT JOIN   wbb" . WCF_N . "_thread thread
            ON          thread.threadID = " . $this->getTableName() . ".threadID";

        if (WCF::getUser()->userID) {
            $join .= "
                LEFT JOIN   wbb" . WCF_N . "_thread_user_status thread_user_status
                ON          thread_user_status.threadID = " . $this->getTableName() . ".threadID
                            AND thread_user_status.userID = " . WCF::getUser()->userID;
        }

        return $join;
    }

    /**
     * @inheritDoc
     */
    public function getTableName(): string
    {
        return 'wbb' . WCF_N . '_post';
    }

    /**
     * @inheritDoc
     */
    public function getIDFieldName(): string
    {
        return $this->getTableName() . '.postID';
    }

    /**
     * @inheritDoc
     */
    public function getAdditionalData(): ?array
    {
        return [
            'findThreads' => $this->findThreads,
            'findAttachments' => $this->findAttachments,
            'findPolls' => $this->findPolls,
            'findFirstPosts' => $this->findFirstPosts,
            'findOfficialPosts' => $this->findOfficialPosts,
            'findThreadsWithBestAnswer' => $this->findThreadsWithBestAnswer,
            'boardID' => $this->boardID,
            'threadID' => $this->threadID,
        ];
    }

    /**
     * @inheritDoc
     */
    public function getConditionBuilder(array $parameters): ?PreparedStatementConditionBuilder
    {
        $this->readParameters($parameters);
        $this->loadSearchableBoardIDs();

        $conditionBuilder = new PreparedStatementConditionBuilder();
        $this->initBoardCondition($conditionBuilder);
        $this->initMiscConditions($conditionBuilder);
        $this->initLanguageCondition($conditionBuilder);

        return $conditionBuilder;
    }

    private function loadSearchableBoardIDs(): void
    {
        if ($this->boardID) {
            if ($board = BoardCache::getInstance()->getBoard($this->boardID)) {
                if ($board->searchable) {
                    $this->searchableBoardIDs[] = $board->boardID;
                }

                foreach ($this->getSubBoardIDs($board->boardID) as $boardID) {
                    $this->searchableBoardIDs[] = $boardID;
                }
            }
        }

        if (empty($this->searchableBoardIDs)) {
            $this->searchableBoardIDs = \array_keys(BoardCache::getInstance()->getBoards());
        }

        $this->searchableBoardIDs = \array_filter($this->searchableBoardIDs, static function ($boardID) {
            $board = BoardCache::getInstance()->getBoard($boardID);
            if (
                $board->isIgnored()
                || !$board->getPermission()
                || !$board->getPermission('canEnterBoard')
                || !$board->getPermission('canReadThread')
                || !$board->searchable
                || ($board->isPrivate && !WCF::getUser()->userID)
            ) {
                return false;
            }

            return true;
        });
        if (\count($this->searchableBoardIDs) == 0) {
            // todo: don't use an exception here
            throw new PermissionDeniedException();
        }
    }

    /**
     * Returns an iterator iterating over the IDs of all recursive sub board of the board with
     * the given ID.
     */
    private function getSubBoardIDs(int $boardID): iterable
    {
        foreach (BoardCache::getInstance()->getChildIDs($boardID) as $childBoardID) {
            yield $childBoardID;

            yield from $this->getSubBoardIDs($childBoardID);
        }
    }

    private function initBoardCondition(PreparedStatementConditionBuilder $conditionBuilder): void
    {
        $boardIDs = $privateBoardIDs = [];
        foreach ($this->searchableBoardIDs as $boardID) {
            $board = BoardCache::getInstance()->getBoard($boardID);
            if ($board->isPrivate && !$board->canReadPrivateThreads()) {
                $privateBoardIDs[] = $board->boardID;
                continue;
            }

            $boardIDs[] = $board->boardID;
        }
        if (empty($privateBoardIDs)) {
            $conditionBuilder->add('thread.boardID IN (?)', [$boardIDs]);
        } elseif (empty($boardIDs)) {
            $conditionBuilder->add(
                '(thread.boardID IN (?) AND thread.userID = ?)',
                [$privateBoardIDs, WCF::getUser()->userID]
            );
        } else {
            $conditionBuilder->add(
                '(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))',
                [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]
            );
        }
    }

    private function initMiscConditions(PreparedStatementConditionBuilder $conditionBuilder): void
    {
        if (!$this->findThreads) {
            $conditionBuilder->add($this->getTableName() . '.isDeleted = 0');
            $conditionBuilder->add($this->getTableName() . '.isDisabled = 0');

            if ($this->findAttachments) {
                $conditionBuilder->add($this->getTableName() . '.attachments > 0');
            }

            if ($this->findFirstPosts) {
                $conditionBuilder->add($this->getTableName() . '.postID = thread.firstPostID');
            }

            if ($this->findPolls) {
                $conditionBuilder->add($this->getTableName() . '.pollID IS NOT NULL');
            }

            if ($this->findOfficialPosts) {
                $conditionBuilder->add($this->getTableName() . '.isOfficial = ?', [1]);
            }
        }

        if ($this->threadID) {
            $conditionBuilder->add($this->getTableName() . '.threadID = ?', [$this->threadID]);
        }

        if ($this->findThreadsWithBestAnswer) {
            $conditionBuilder->add('thread.bestAnswerPostID IS NOT NULL');
        }

        if (WCF::getUser()->userID) {
            // filter ignored threads
            $conditionBuilder->add(
                "(thread_user_status.status <> ? OR thread_user_status.status IS NULL)",
                [ThreadStatusHandler::SUBSCRIPTION_MODE_IGNORING]
            );
        }
    }

    private function initLanguageCondition(PreparedStatementConditionBuilder $conditionBuilder): void
    {
        if (LanguageFactory::getInstance()->multilingualismEnabled() && \count(WCF::getUser()->getLanguageIDs())) {
            $conditionBuilder->add(
                '(thread.languageID IN (?) OR thread.languageID IS NULL)',
                [WCF::getUser()->getLanguageIDs()]
            );
        }
    }

    /**
     * @inheritDoc
     */
    public function assignVariables(): void
    {
        $boardNodeList = new SearchableBoardNodeList();
        $boardNodeList->readNodeTree();

        $thread = null;
        if (!empty($_REQUEST['threadID'])) {
            $thread = new Thread(\intval($_REQUEST['threadID']));
            if (!$thread->threadID || !$thread->canRead()) {
                $thread = null;
            }
        }

        WCF::getTPL()->assign([
            'boardNodeList' => $boardNodeList->getNodeList(),
            'searchedThread' => $thread,
        ]);
    }

    /**
     * @inheritDoc
     */
    public function getFetchObjectsQuery(?PreparedStatementConditionBuilder $additionalConditions = null): string
    {
        if (!$this->findThreads) {
            return '';
        }

        $join = '';
        if (WCF::getUser()->userID) {
            $join = "
                LEFT JOIN   wbb" . WCF_N . "_thread_user_status thread_user_status
                ON          thread_user_status.threadID = thread.threadID
                            AND thread_user_status.userID = " . WCF::getUser()->userID;
        }

        return "SELECT      thread.threadID AS objectID, thread.topic AS subject,
                            thread.time, thread.username,
                            'com.woltlab.wbb.post' AS objectType
                FROM        wbb" . WCF_N . "_thread thread
                " . $join . "
                INNER JOIN  (
                                SELECT      DISTINCT post.threadID
                                FROM        wbb" . WCF_N . "_post post
                                INNER JOIN  ({WCF_SEARCH_INNER_JOIN}) search_index
                                ON          post.postID = search_index.objectID
                                WHERE       post.isDeleted = 0
                                        AND post.isDisabled = 0
                                        " . ($this->findAttachments ? "AND post.attachments > 0" : '') . "
                                        " . ($this->findPolls ? "AND post.pollID IS NOT NULL" : '') . "
                                        " . ($this->findOfficialPosts ? "AND post.isOfficial = 1" : '') . "
                                        " . ($this->findFirstPosts ? "AND post.postID IN (SELECT firstPostID FROM wbb" . WCF_N . "_thread WHERE threadID = post.threadID)" : '') . "
                            ) AS subselect
                ON          thread.threadID = subselect.threadID
                " . ($additionalConditions !== null ? $additionalConditions : '');
    }

    /**
     * @inheritDoc
     */
    public function getResultListTemplateName(): string
    {
        if ($this->findThreads) {
            return 'searchResultThreadList';
        }

        return '';
    }

    /**
     * @inheritDoc
     */
    public function getCustomSortField(string $sortField): string
    {
        if ($this->findThreads && $sortField === 'relevance') {
            return 'time';
        }

        return '';
    }

    private function readParameters(array $parameters): void
    {
        $this->findThreads = 0;

        if (!empty($parameters['boardID'])) {
            $this->boardID = \intval($parameters['boardID']);
        }

        if (!empty($parameters['findAttachments'])) {
            $this->findAttachments = 1;
        }

        if (!empty($parameters['findPolls'])) {
            $this->findPolls = 1;
        }

        if (!empty($parameters['findFirstPosts'])) {
            $this->findFirstPosts = 1;
        }

        if (!empty($parameters['findOfficialPosts'])) {
            $this->findOfficialPosts = 1;
        }

        if (!empty($parameters['findThreadsWithBestAnswer'])) {
            $this->findThreadsWithBestAnswer = 1;
        }

        if (!empty($parameters['findThreads'])) {
            $this->findThreads = 1;
        }

        if (!empty($parameters['threadID'])) {
            $this->threadID = \intval($parameters['threadID']);

            // make sure that the results are never grouped by thread
            $this->findThreads = 0;
        }
    }
}
