/**
 * Inline editor for posts.
 *
 * @author  Alexander Ebert
 * @copyright  2001-2019 WoltLab GmbH
 * @license  WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module  WoltLabSuite/Forum/Ui/Post/InlineEditor
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Core", "WoltLabSuite/Core/Language", "WoltLabSuite/Core/Ui/Confirmation", "WoltLabSuite/Core/Ui/Message/InlineEditor", "./Enable"], function (require, exports, tslib_1, Core, Language, UiConfirmation, InlineEditor_1, UiPostEnable) {
    "use strict";
    Core = tslib_1.__importStar(Core);
    Language = tslib_1.__importStar(Language);
    UiConfirmation = tslib_1.__importStar(UiConfirmation);
    InlineEditor_1 = tslib_1.__importDefault(InlineEditor_1);
    UiPostEnable = tslib_1.__importStar(UiPostEnable);
    class UiPostInlineEditor extends InlineEditor_1.default {
        constructor(threadId, postManager, opts) {
            const options = Core.extend({
                disableEdit: false,
            }, opts);
            super({
                canEditInline: options.canEditInline,
                className: "wbb\\data\\post\\PostAction",
                containerId: threadId.toString(),
                dropdownIdentifier: "wbbPost",
                editorPrefix: "postEditor",
                quoteManager: options.quoteManager || null,
            });
            this.disableEdit = options.disableEdit;
            this.postManager = postManager;
            this.threadId = threadId;
        }
        _showMessage(data) {
            const activeElement = this._activeElement;
            super._showMessage(data);
            Object.entries(data.returnValues.notes).forEach(([className, html]) => {
                this.postManager.setNote(this._getObjectId(activeElement), className, html);
            });
            const elementData = this._elements.get(activeElement);
            if (data.returnValues.threadFormOptions) {
                const threadFormOptions = elementData.messageBody.querySelector(".threadFormOptions");
                if (threadFormOptions !== null) {
                    threadFormOptions.innerHTML = data.returnValues.threadFormOptions;
                }
                else {
                    const div = document.createElement("div");
                    div.classList.add("threadFormOptions");
                    div.innerHTML = data.returnValues.threadFormOptions;
                    elementData.messageBody.insertAdjacentElement("afterbegin", div);
                }
            }
            else {
                const threadFormOptions = elementData.messageBody.querySelector(".threadFormOptions");
                threadFormOptions === null || threadFormOptions === void 0 ? void 0 : threadFormOptions.remove();
            }
            if (data.returnValues.isOfficial !== undefined) {
                const messageStatus = elementData.messageHeader.querySelector(".messageStatus");
                const iconOfficial = messageStatus.querySelector(".jsIconOfficial");
                const message = elementData.messageBody.closest(".message").parentElement;
                if (data.returnValues.isOfficial) {
                    if (!iconOfficial) {
                        const listItem = document.createElement("li");
                        listItem.innerHTML = `<span class="badge label blue jsIconOfficial">${Language.get("wbb.post.official")}</span>`;
                        messageStatus.appendChild(listItem);
                    }
                    message.classList.add("wbbPostOfficial");
                }
                else {
                    iconOfficial === null || iconOfficial === void 0 ? void 0 : iconOfficial.remove();
                    message.classList.remove("wbbPostOfficial");
                }
            }
        }
        /**
         * @see WoltLabSuite/Core/Ui/Message/InlineEditor._dropdownGetItems()
         */
        _dropdownGetItems() {
            const prefix = "wbb.post.edit.";
            return [
                { item: "enable", label: prefix + "enable" },
                { item: "disable", label: prefix + "disable" },
                { item: "close", label: prefix + "close" },
                { item: "open", label: prefix + "open" },
                { item: "trash", label: prefix + "delete" },
                { item: "restore", label: prefix + "restore" },
                { item: "deleteCompletely", label: prefix + "deleteCompletely" },
                { item: "divider" },
                { item: "editItem", label: "wbb.post.edit" },
            ];
        }
        /**
         * @see WoltLabSuite/Core/Ui/Message/InlineEditor._dropdownOpen()
         */
        _dropdownOpen() {
            const objectId = this._getObjectId(this._activeDropdownElement);
            const canClose = this.postManager.getPermission(objectId, "close");
            const canEnable = this.postManager.getPermission(objectId, "enable");
            const isClosed = this.postManager.getPropertyValue(objectId, "is-closed", true);
            const isDeleted = this.postManager.getPropertyValue(objectId, "is-deleted", true);
            const isDisabled = this.postManager.getPropertyValue(objectId, "is-disabled", true);
            return {
                close: canClose && !isClosed,
                trash: this.postManager.getPermission(objectId, "delete") && !isDeleted,
                deleteCompletely: this.postManager.getPermission(objectId, "delete-completely") && isDeleted,
                disable: canEnable && !isDisabled && !isDeleted,
                enable: canEnable && isDisabled && !isDeleted,
                open: canClose && isClosed,
                restore: this.postManager.getPermission(objectId, "restore") && isDeleted,
                editItem: !this.disableEdit,
            };
        }
        /**
         * @see WoltLabSuite/Core/Ui/Message/InlineEditor._dropdownSelect()
         */
        _dropdownSelect(item) {
            const objectId = ~~this._getObjectId(this._activeDropdownElement);
            const callback = (parameters = {}) => {
                this.postManager.update(objectId.toString(), item, parameters);
            };
            switch (item) {
                case "deleteCompletely":
                    UiConfirmation.show({
                        confirm: callback,
                        message: Language.get("wbb.post.edit.delete.confirmMessage"),
                    });
                    break;
                case "enable":
                    UiPostEnable.init(objectId, callback);
                    break;
                case "trash":
                    UiConfirmation.show({
                        confirm() {
                            const trashReason = document.getElementById("wbbPostTrashReason");
                            callback({
                                reason: trashReason.value.trim(),
                            });
                        },
                        message: Language.get("wbb.post.edit.trash.confirmMessage"),
                        template: `
<dl>
  <dt>
    <label for="wbbPostTrashReason">${Language.get("wbb.post.edit.trash.reason")}</label>
  </dt>
  <dd>
    <textarea id="wbbPostTrashReason" cols="40" rows="3"></textarea>
  </dd>
</dl>`,
                    });
                    break;
                default:
                    callback();
                    break;
            }
        }
        /**
         * @see WoltLabSuite/Core/Ui/Message/InlineEditor._getHash()
         */
        _getHash(objectId) {
            return "#post" + objectId;
        }
    }
    Core.enableLegacyInheritance(UiPostInlineEditor);
    return UiPostInlineEditor;
});
