<?php

namespace wbb\data\post;

use wbb\data\board\Board;
use wbb\data\thread\Thread;
use wcf\system\clipboard\ClipboardHandler;

/**
 * Represents a list of deleted posts.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Posts
 */
class DeletedPostList extends ViewablePostList
{
    /**
     * @inheritDoc
     */
    public $sqlOrderBy = 'post.time DESC, post.postID DESC';

    /**
     * @inheritDoc
     */
    protected $attachmentLoading = false;

    /**
     * @inheritDoc
     */
    protected $pollLoading = false;

    /**
     * @inheritDoc
     */
    public function __construct()
    {
        $this->sqlConditionJoins .= " JOIN wbb" . WCF_N . "_thread thread";
        $this->sqlJoins .= "
            LEFT JOIN   wbb" . WCF_N . "_thread thread
            ON          thread.threadID = post.threadID";

        parent::__construct();

        if (!empty($this->sqlSelects)) {
            $this->sqlSelects .= ',';
        }
        $this->sqlSelects .= "thread.topic, thread.boardID";
        $this->getConditionBuilder()->add('thread.threadID = post.threadID');
        $this->getConditionBuilder()->add('post.isDeleted = ?', [1]);
        $boardIDs = \array_intersect(
            Board::getAccessibleModeratorBoardIDs(['canReadDeletedPost']),
            Board::getAccessibleBoardIDs(['canViewBoard', 'canEnterBoard', 'canReadThread'])
        );
        if (!empty($boardIDs)) {
            $this->getConditionBuilder()->add('thread.boardID IN (?)', [$boardIDs]);
        } else {
            $this->getConditionBuilder()->add('1 = 0');
        }
    }

    /**
     * @inheritDoc
     */
    public function readObjects()
    {
        parent::readObjects();

        $postIDs = [];
        foreach ($this->getObjects() as $post) {
            /** @noinspection PhpUndefinedFieldInspection */
            $post->setThread(new Thread(null, [
                'topic' => $post->topic,
                'threadID' => $post->threadID,
                'boardID' => $post->boardID,
            ]));

            $postIDs[] = $post->postID;
        }
    }

    /**
     * Returns the number of marked items.
     *
     * @return  int
     */
    public function getMarkedItems()
    {
        return ClipboardHandler::getInstance()->hasMarkedItems(
            ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.post')
        );
    }
}
