<?php

namespace wbb\data\board;

use wbb\system\board\status\BoardStatusHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\WCF;

/**
 * Represents a list of watched board nodes.
 *
 * @author  Joshua Ruesweg, Matthias Schmidt
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Board
 * @since   5.0
 */
class WatchedBoardNodeList extends DetailedBoardNodeList
{
    /**
     * ids of watched boards
     * @var int[]
     */
    protected $watchedBoardIDs = [];

    /**
     * @inheritDoc
     */
    public function __construct($parentID = null, $startDepth = 0, array $filter = [])
    {
        parent::__construct($parentID, $startDepth, $filter);

        $conditionBuilder = new PreparedStatementConditionBuilder();
        $conditionBuilder->add('userID = ?', [WCF::getUser()->userID]);
        $conditionBuilder->add("status IN (?)", [
            [BoardStatusHandler::SUBSCRIPTION_MODE_THREADSANDPOSTS, BoardStatusHandler::SUBSCRIPTION_MODE_THREADSONLY],
        ]);
        $sql = "SELECT  boardID
                FROM    wbb1_board_user_status
                " . $conditionBuilder;
        $statement = WCF::getDB()->prepare($sql);
        $statement->execute($conditionBuilder->getParameters());
        $this->watchedBoardIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
    }

    /**
     * @inheritDoc
     */
    protected function generateNodeTree($parentID, ?BoardNode $parentNode = null, array $filter = [])
    {
        $nodes = [];

        // apply filter
        if (\count($filter)) {
            $this->watchedBoardIDs = \array_intersect($this->watchedBoardIDs, $filter);
        }

        foreach ($this->watchedBoardIDs as $boardID) {
            $board = BoardCache::getInstance()->getBoard($boardID);
            if ($this->isVisible($board)) {
                $node = new $this->nodeClassname(
                    $parentNode,
                    $board,
                    $parentNode !== null ? $parentNode->getDepth() + 1 : 0
                );
                $nodes[] = $node;
            }
        }

        return $nodes;
    }

    /**
     * @inheritDoc
     */
    protected function isVisible(Board $board)
    {
        // do not check for ignored boards
        return RestrictedBoardNodeList::isVisible($board);
    }
}
