<?php

namespace wbb\data\post;

use wbb\data\thread\ViewableThreadList;
use wbb\system\cache\runtime\PostRuntimeCache;
use wcf\data\like\IRestrictedLikeObjectTypeProvider;
use wcf\data\like\object\ILikeObject;
use wcf\system\like\IViewableLikeProvider;
use wcf\system\WCF;

/**
 * Object type provider for likeable posts.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Post
 *
 * @method  LikeablePost        getObjectByID($objectID)
 */
class LikeablePostProvider extends PostProvider implements IRestrictedLikeObjectTypeProvider, IViewableLikeProvider
{
    /**
     * @inheritDoc
     */
    public $decoratorClassName = LikeablePost::class;

    /**
     * @inheritDoc
     */
    public function checkPermissions(ILikeObject $object)
    {
        return $this->canLike($object);
    }

    /**
     * @inheritDoc
     */
    public function canLike(ILikeObject $object)
    {
        /** @var LikeablePost $object */
        return $this->canViewLikes($object) && $object->getThread()->getBoard()->getPermission('canLikePost');
    }

    /**
     * @inheritDoc
     */
    public function canViewLikes(ILikeObject $object)
    {
        /** @var LikeablePost $object */
        return $object->postID && $object->canRead();
    }

    /**
     * @inheritDoc
     */
    public function getObjectsByIDs(array $objectIDs)
    {
        return \array_map(static function (Post $post) {
            return new LikeablePost($post);
        }, PostRuntimeCache::getInstance()->getObjects($objectIDs));
    }

    /**
     * @inheritDoc
     */
    public function prepare(array $likes)
    {
        $postIDs = [];
        foreach ($likes as $like) {
            $postIDs[] = $like->objectID;
        }

        // fetch posts
        $postList = new PostList();
        $postList->setObjectIDs($postIDs);
        $postList->readObjects();
        $posts = $postList->getObjects();

        // fetch threads
        $threadIDs = $threads = [];
        foreach ($posts as $post) {
            $threadIDs[] = $post->threadID;
        }
        if (!empty($threadIDs)) {
            $threadList = new ViewableThreadList();
            $threadList->setObjectIDs($threadIDs);
            $threadList->readObjects();
            $threads = $threadList->getObjects();
        }

        // set message
        foreach ($likes as $like) {
            if (isset($posts[$like->objectID])) {
                $post = $posts[$like->objectID];
                $thread = $threads[$post->threadID];
                $post->setThread($thread->getDecoratedObject());

                // check permissions
                if (!$post->canRead()) {
                    continue;
                }
                $like->setIsAccessible();

                // short output
                $text = WCF::getLanguage()->getDynamicVariable('wcf.like.title.com.woltlab.wbb.likeablePost', [
                    'post' => $post,
                    'thread' => $threads[$post->threadID],
                    'reaction' => $like,
                    // @deprecated 5.3 Use `$reaction` instead
                    'like' => $like,
                ]);
                $like->setTitle($text);

                // output
                $like->setDescription($post->getExcerpt());
            }
        }
    }
}
