<?php

namespace wbb\form;

use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\board\RestrictedBoardNodeList;
use wbb\data\post\Post;
use wbb\data\post\PostEditor;
use wbb\data\thread\form\ThreadForm;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadAction;
use wbb\data\thread\ThreadEditor;
use wbb\page\IBoardPage;
use wbb\system\label\object\ThreadLabelObjectHandler;
use wbb\system\option\ThreadFormOptionHandler;
use wbb\system\WBBCore;
use wcf\data\label\group\ViewableLabelGroup;
use wcf\data\user\User;
use wcf\form\MessageForm;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\flood\FloodControl;
use wcf\system\label\LabelHandler;
use wcf\system\language\LanguageFactory;
use wcf\system\message\quote\MessageQuoteManager;
use wcf\system\poll\PollManager;
use wcf\system\WCF;
use wcf\util\ArrayUtil;
use wcf\util\HeaderUtil;
use wcf\util\StringUtil;
use wcf\util\UserRegistrationUtil;

/**
 * Shows the new thread form.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Form
 */
class ThreadAddForm extends MessageForm implements IBoardPage
{
    /**
     * @inheritDoc
     */
    public $attachmentObjectType = 'com.woltlab.wbb.post';

    /**
     * board id
     * @var int
     */
    public $boardID = 0;

    /**
     * list of board ids for announcements
     * @var int[]
     */
    public $boardIDs = [];

    /**
     * board object
     * @var Board
     */
    public $board;

    /**
     * @var ThreadFormOptionHandler
     */
    public $optionHandler;

    /**
     * board node list
     * @var RestrictedBoardNodeList
     */
    public $boardList;

    /**
     * close thread after post
     * @var bool
     */
    public $closeThread = false;

    /**
     * disable thread after post
     * @var bool
     */
    public $disableThread = false;

    /**
     * publication date (ISO 8601)
     * @var string
     */
    public $enableTime = '';

    /**
     * publication date object
     * @var \DateTime
     */
    public $enableTimeObj;

    /**
     * subscribe thread
     * @var bool
     */
    public $subscribeThread = false;

    /**
     * @var bool
     * @since 5.4
     */
    public $markAsOfficial = false;

    /**
     * @inheritDoc
     */
    public $enableMultilingualism = true;

    /**
     * label group list
     * @var ViewableLabelGroup[]
     */
    public $labelGroups = [];

    /**
     * list of label ids
     * @var int[]
     */
    public $labelIDs = [];

    /**
     * @inheritDoc
     */
    public $messageObjectType = 'com.woltlab.wbb.post';

    /**
     * username
     * @var string
     */
    public $username = '';

    /**
     * tags
     * @var array
     */
    public $tags = [];

    /**
     * thread type (0 = default, 1 = sticky, 2 = announcement)
     * @var int
     */
    public $type = 0;

    /**
     * additional fields for the first post
     * @var mixed[][]
     */
    public $additionalPostFields = [];

    /**
     * minimum number of characters in message text
     * @var int
     */
    public $minCharLength = WBB_THREAD_MIN_CHAR_LENGTH;

    /**
     * minimum number of words in message text
     * @var int
     */
    public $minWordCount = WBB_THREAD_MIN_WORD_COUNT;

    /**
     * @var ThreadForm
     */
    public $threadForm;

    /**
     * @inheritDoc
     */
    public function readParameters()
    {
        parent::readParameters();

        if (isset($_REQUEST['id'])) {
            $this->boardID = \intval($_REQUEST['id']);
        }
        $this->board = BoardCache::getInstance()->getBoard($this->boardID);
        if ($this->board === null || !$this->board->isBoard() || $this->board->isClosed) {
            throw new IllegalLinkException();
        }

        // The tmpHash is already handled in MessageForm::readParameters().
        // To generate deterministic tmpHashes we need to duplicate some of the logic.
        if (isset($_REQUEST['tmpHash'])) {
            $this->tmpHash = $_REQUEST['tmpHash'];
        } else {
            $this->tmpHash = \sha1(\implode("\0", [
                // Use class name + board ID to match the autosave scoping.
                self::class,
                $this->board->boardID,
                // Bind the tmpHash to the current session to make it unguessable.
                WCF::getSession()->sessionID,
            ]));
        }

        // check permissions
        $this->board->checkPermission(['canViewBoard', 'canEnterBoard', 'canStartThread']);

        // check flood control
        Post::enforceFloodControl();

        // set attachment parent id
        $this->attachmentParentObjectID = $this->board->boardID;

        // polls
        if ($this->canUsePoll()) {
            PollManager::getInstance()->setObject('com.woltlab.wbb.post', 0);
        }

        // init board style
        $this->board->initStyle();

        // get max text length
        $this->maxTextLength = WCF::getSession()->getPermission('user.board.maxPostLength');

        // quotes
        MessageQuoteManager::getInstance()->readParameters();

        // labels
        ThreadLabelObjectHandler::getInstance()->setBoardID($this->board->boardID);

        $this->optionHandler = new ThreadFormOptionHandler(false);
        $this->optionHandler->setBoardID($this->board->boardID);
        $this->optionHandler->init();
    }

    /**
     * @inheritDoc
     */
    public function readFormParameters()
    {
        parent::readFormParameters();

        if (isset($_POST['boardIDs']) && \is_array($_POST['boardIDs'])) {
            $this->boardIDs = ArrayUtil::toIntegerArray($_POST['boardIDs']);
        }
        if (isset($_POST['labelIDs']) && \is_array($_POST['labelIDs'])) {
            $this->labelIDs = $_POST['labelIDs'];
        }
        if (isset($_POST['tags']) && \is_array($_POST['tags'])) {
            $this->tags = ArrayUtil::trim($_POST['tags']);
        }
        if (isset($_POST['type'])) {
            $this->type = \intval($_POST['type']);
        }
        if (isset($_POST['username'])) {
            $this->username = StringUtil::trim($_POST['username']);
        }
        if (!empty($_POST['enableTime'])) {
            $this->enableTime = $_POST['enableTime'];
            $this->enableTimeObj = \DateTime::createFromFormat('Y-m-d\TH:i:sP', $this->enableTime);
        }

        // settings
        if (isset($_POST['closeThread']) && $this->board->getModeratorPermission('canCloseThread')) {
            $this->closeThread = true;
        }
        if (isset($_POST['disableThread']) && $this->board->getModeratorPermission('canEnableThread')) {
            $this->disableThread = true;
        }
        if (isset($_POST['subscribeThread']) && WCF::getUser()->userID) {
            $this->subscribeThread = true;
        }
        if (isset($_POST['markAsOfficial']) && WCF::getSession()->getPermission('mod.board.canMarkPostOfficial')) {
            $this->markAsOfficial = true;
        }

        // polls
        if (WCF::getSession()->getPermission('user.board.canStartPoll')) {
            PollManager::getInstance()->readFormParameters();
        }

        // quotes
        MessageQuoteManager::getInstance()->readFormParameters();

        $this->optionHandler->readUserInput($_POST);
    }

    /**
     * @inheritDoc
     */
    public function validate()
    {
        // validate file options; call this first so that values are always read
        $optionHandlerErrors = $this->optionHandler->validate();

        if (!empty($optionHandlerErrors)) {
            throw new UserInputException('options', $optionHandlerErrors);
        }

        parent::validate();

        $this->validateUsername();
        $this->validateLabelIDs();
        $this->validateType();

        if (!empty($this->enableTime)) {
            if (!$this->enableTimeObj || $this->enableTimeObj->getTimestamp() < TIME_NOW) {
                throw new UserInputException('enableTime', 'invalid');
            }
        }

        // polls
        if ($this->canUsePoll()) {
            PollManager::getInstance()->validate();
        }
    }

    /**
     * @inheritDoc
     */
    protected function validateText()
    {
        parent::validateText();

        $message = $this->htmlInputProcessor->getTextContent();
        if ($this->minCharLength && (\mb_strlen($message) < $this->minCharLength)) {
            throw new UserInputException('text', 'minCharLength');
        }

        if ($this->minWordCount && (\count(\explode(' ', $message)) < $this->minWordCount)) {
            throw new UserInputException('text', 'minWordCount');
        }
    }

    /**
     * Validates the username.
     */
    protected function validateUsername()
    {
        // only for guests
        if (!WCF::getUser()->userID) {
            if (empty($this->username)) {
                throw new UserInputException('username');
            }
            if (!UserRegistrationUtil::isValidUsername($this->username)) {
                throw new UserInputException('username', 'invalid');
            }
            if (User::getUserByUsername($this->username)->userID) {
                throw new UserInputException('username', 'notUnique');
            }

            WCF::getSession()->register('username', $this->username);
        }
    }

    /**
     * Validates label ids.
     */
    protected function validateLabelIDs()
    {
        $validationResult = ThreadLabelObjectHandler::getInstance()->validateLabelIDs(
            $this->labelIDs,
            'canSetLabel',
            false
        );
        if (!empty($validationResult[0])) {
            throw new UserInputException('labelIDs');
        }

        if (!empty($validationResult)) {
            throw new UserInputException('label', $validationResult);
        }
    }

    /**
     * Validates thread type.
     */
    protected function validateType()
    {
        switch ($this->type) {
            case Thread::TYPE_STICKY:
                // validate permissions
                if (!$this->board->getModeratorPermission('canPinThread')) {
                    throw new PermissionDeniedException();
                }

                // reset board ids
                $this->boardIDs = [];
                break;

            case Thread::TYPE_ANNOUNCEMENT:
                // validate permissions
                if (!$this->board->getModeratorPermission('canStartAnnouncement')) {
                    throw new PermissionDeniedException();
                }

                // validate board ids
                foreach ($this->boardIDs as $key => $boardID) {
                    $board = BoardCache::getInstance()->getBoard($boardID);
                    if (
                        $board === null
                        || !$board->isBoard()
                        || !$board->getModeratorPermission('canStartAnnouncement')
                    ) {
                        unset($this->boardIDs[$key]);
                    } else {
                        $board->checkPermission(['canViewBoard', 'canEnterBoard']);
                    }
                }

                if (empty($this->boardIDs)) {
                    $this->boardIDs = [$this->board->boardID];
                }
                break;

            default:
            case Thread::TYPE_DEFAULT:
                $this->type = 0;

                // reset board ids
                $this->boardIDs = [];
                break;
        }
    }

    /**
     * @inheritDoc
     */
    public function save()
    {
        parent::save();

        // save thread
        $data = \array_merge($this->additionalFields, [
            'boardID' => $this->boardID,
            'languageID' => $this->languageID,
            'topic' => $this->subject,
            'time' => TIME_NOW,
            'userID' => WCF::getUser()->userID ?: null,
            'username' => WCF::getUser()->userID ? WCF::getUser()->username : $this->username,
            'hasLabels' => !empty($this->labelIDs) ? 1 : 0,
        ]);
        if ($this->closeThread) {
            $data['isClosed'] = 1;
        }
        if ($this->disableThread || !$this->board->getPermission('canStartThreadWithoutModeration')) {
            $data['isDisabled'] = 1;
        }

        $threadData = [
            'data' => $data,
            'board' => $this->board,
            'attachmentHandler' => $this->attachmentHandler,
            'htmlInputProcessor' => $this->htmlInputProcessor,
            'postData' => \array_merge(
                $this->additionalPostFields,
                [
                    'enableTime' => $this->enableTimeObj ? $this->enableTimeObj->getTimestamp() : 0,
                    'isOfficial' => $this->markAsOfficial ? 1 : 0,
                ]
            ),
            'tags' => [],
            'subscribeThread' => $this->subscribeThread,
            'optionHandler' => $this->optionHandler,
        ];

        // handle thread type
        switch ($this->type) {
            case Thread::TYPE_STICKY:
                $threadData['data']['isSticky'] = 1;
                break;

            case Thread::TYPE_ANNOUNCEMENT:
                $threadData['data']['isAnnouncement'] = 1;
                $threadData['announcementBoardIDs'] = $this->boardIDs;
                break;
        }

        if (
            MODULE_TAGGING
            && WBB_THREAD_ENABLE_TAGS
            && WCF::getSession()->getPermission('user.tag.canViewTag')
            && $this->board->getPermission('canSetTags')
        ) {
            $threadData['tags'] = $this->tags;
        }
        $this->objectAction = new ThreadAction([], 'create', $threadData);

        /** @var Thread $thread */
        $thread = $this->objectAction->executeAction()['returnValues'];

        // save labels
        if (!empty($this->labelIDs)) {
            ThreadLabelObjectHandler::getInstance()->setLabels($this->labelIDs, $thread->threadID);
        }

        // save polls
        if ($this->canUsePoll()) {
            $thread = new Thread($thread->threadID);
            $pollID = PollManager::getInstance()->save($thread->firstPostID);
            $addedPoll = false;
            if ($pollID) {
                $postEditor = new PostEditor(new Post($thread->firstPostID));
                $postEditor->update([
                    'pollID' => $pollID,
                ]);

                $addedPoll = true;
            }

            if ($addedPoll) {
                $threadEditor = new ThreadEditor($thread);
                $threadEditor->updateCounters([
                    'polls' => 1,
                ]);
            }
        }

        MessageQuoteManager::getInstance()->saved();

        FloodControl::getInstance()->registerContent('com.woltlab.wbb.post');

        $this->saved();

        if ($thread->isDisabled && !WCF::getUser()->userID) {
            HeaderUtil::delayedRedirect(
                $this->board->getLink(),
                WCF::getLanguage()->getDynamicVariable('wbb.thread.moderation.redirect'),
                30
            );
        } else {
            HeaderUtil::redirect($thread->getLink());
        }

        exit;
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        parent::readData();

        $this->boardList = new RestrictedBoardNodeList();
        $this->boardList->readNodeTree();
        $labelGroupIDs = BoardCache::getInstance()->getLabelGroupIDs($this->board->boardID);
        if (!empty($labelGroupIDs)) {
            $this->labelGroups = LabelHandler::getInstance()->getLabelGroups($labelGroupIDs);
        }
        if ($this->board->formID) {
            $this->threadForm = new ThreadForm($this->board->formID);
        }

        if (empty($_POST)) {
            // default values
            $this->username = WCF::getSession()->getVar('username');
            /** @noinspection PhpUndefinedFieldInspection */
            if (WCF::getUser()->watchThreadOnReply) {
                $this->subscribeThread = true;
            }
            if (WCF::getSession()->getPermission('mod.board.canMarkPostOfficial')) {
                $this->markAsOfficial = !!WBB_OFFICIAL_POST_DEFAULT;
            }
            $this->boardIDs = [$this->boardID];

            // multilingualism
            if (!empty($this->availableContentLanguages)) {
                if (!$this->languageID) {
                    $language = LanguageFactory::getInstance()->getUserLanguage();
                    $this->languageID = $language->languageID;
                }

                if (!isset($this->availableContentLanguages[$this->languageID])) {
                    $languageIDs = \array_keys($this->availableContentLanguages);
                    $this->languageID = \array_shift($languageIDs);
                }
            }

            $this->optionHandler->readData();
        }

        // add breadcrumbs
        WBBCore::getInstance()->setLocation($this->board->getParentBoards(), $this->board);
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        if ($this->canUsePoll()) {
            PollManager::getInstance()->assignVariables();
        }
        MessageQuoteManager::getInstance()->assignVariables();

        WCF::getTPL()->assign([
            'boardID' => $this->boardID,
            'boardIDs' => $this->boardIDs,
            'board' => $this->board,
            'boardNodeList' => $this->boardList->getNodeList(),
            'closeThread' => $this->closeThread,
            'disableThread' => $this->disableThread,
            'subscribeThread' => $this->subscribeThread,
            'markAsOfficial' => $this->markAsOfficial,
            'labelGroups' => $this->labelGroups,
            'labelIDs' => $this->labelIDs,
            'username' => $this->username,
            'tags' => $this->tags,
            'type' => $this->type,
            'minCharLength' => $this->minCharLength,
            'minWordCount' => $this->minWordCount,
            'enableTime' => $this->enableTime,
            'options' => $this->optionHandler->getOptions(),
            'threadForm' => $this->threadForm,
        ]);
    }

    /**
     * Returns true if current user may use polls.
     *
     * @return  bool
     */
    protected function canUsePoll()
    {
        if (!MODULE_POLL || !$this->board->getPermission('canStartPoll')) {
            return false;
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function getBoard()
    {
        return $this->board;
    }
}
