<?php

namespace wbb\page;

use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\board\BoardEditor;
use wbb\data\board\BoardTagCloud;
use wbb\data\board\DetailedBoardNodeList;
use wbb\data\thread\BoardThreadList;
use wbb\system\WBBCore;
use wcf\data\label\group\ViewableLabelGroup;
use wcf\data\language\Language;
use wcf\data\user\group\UserGroup;
use wcf\data\user\ignore\UserIgnore;
use wcf\data\user\UserProfile;
use wcf\page\SortablePage;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\clipboard\ClipboardHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\label\LabelHandler;
use wcf\system\language\LanguageFactory;
use wcf\system\MetaTagHandler;
use wcf\system\request\LinkHandler;
use wcf\system\user\UserProfileHandler;
use wcf\system\WCF;
use wcf\util\HeaderUtil;
use wcf\util\StringUtil;

/**
 * Shows the board page.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Page
 *
 * @property    BoardThreadList|null    $objectList
 */
class BoardPage extends SortablePage implements IBoardPage
{
    /**
     * @inheritDoc
     */
    public $defaultSortField = WBB_BOARD_DEFAULT_SORT_FIELD;

    /**
     * @inheritDoc
     */
    public $defaultSortOrder = WBB_BOARD_DEFAULT_SORT_ORDER;

    /**
     * @inheritDoc
     */
    public $validSortFields = ['topic', 'username', 'time', 'replies', 'cumulativeLikes', 'views', 'lastPostTime'];

    /**
     * @inheritDoc
     */
    public $itemsPerPage = WBB_BOARD_THREADS_PER_PAGE;

    /**
     * list of available content languages
     * @var Language[]
     */
    public $availableLanguages = [];

    /**
     * board id
     * @var int
     */
    public $boardID = 0;

    /**
     * board object
     * @var Board
     */
    public $board;

    /**
     * board node list
     * @var DetailedBoardNodeList
     */
    public $boardNodeList;

    /**
     * date filter
     * @var int
     */
    public $filterByDate = WBB_BOARD_DEFAULT_DAYS_PRUNE;

    /**
     * status filter
     * @var string
     */
    public $filterByStatus = '';

    /**
     * label filter
     * @var int[]
     */
    public $labelIDs = [];

    /**
     * list of available label groups
     * @var ViewableLabelGroup[]
     */
    public $labelGroups = [];

    /**
     * language filter
     * @var int
     */
    public $languageID = 0;

    /**
     * list of moderator user profiles
     * @var UserProfile[]
     */
    public $userModerators = [];

    /**
     * list of moderator groups
     * @var UserGroup[]
     */
    public $groupModerators = [];

    /**
     * tag id
     * @var int
     */
    public $tagID = 0;

    /**
     * query parameters for tags
     * @var string
     */
    public $tagLinkParameters = '';

    /**
     * true if view is affected by active filters
     * @var bool
     */
    public $hasActiveFilter = false;

    /**
     * @inheritDoc
     */
    public function readParameters()
    {
        parent::readParameters();

        if (isset($_REQUEST['id'])) {
            $this->boardID = \intval($_REQUEST['id']);
        }
        $this->board = BoardCache::getInstance()->getBoard($this->boardID);
        if ($this->board === null) {
            throw new IllegalLinkException();
        }

        // check permissions
        if (!$this->board->canEnter()) {
            throw new PermissionDeniedException();
        }

        // redirect to external url if given
        if ($this->board->isExternalLink()) {
            if (!WCF::getSession()->spiderID) {
                $boardEditor = new BoardEditor($this->board);
                $boardEditor->updateCounters(['clicks' => 1]);
                BoardEditor::resetDataCache();
            }

            // do redirect
            HeaderUtil::redirect($this->board->externalURL);

            exit;
        }

        // init board style
        $this->board->initStyle();

        // threads per page
        if ($this->board->threadsPerPage) {
            $this->itemsPerPage = $this->board->threadsPerPage;
        }
        /** @noinspection PhpUndefinedFieldInspection */
        if (WCF::getUser()->threadsPerPage) {
            /** @noinspection PhpUndefinedFieldInspection */
            $this->itemsPerPage = WCF::getUser()->threadsPerPage;
        }

        // get sorting values
        if ($this->board->sortField) {
            $this->defaultSortField = $this->board->sortField;
        }
        if ($this->board->sortOrder) {
            $this->defaultSortOrder = $this->board->sortOrder;
        }

        // date filter
        if ($this->board->daysPrune) {
            $this->filterByDate = $this->board->daysPrune;
        }
        if (isset($_REQUEST['filterByDate'])) {
            $this->filterByDate = \intval($_REQUEST['filterByDate']);
        }

        // status filter
        if (isset($_REQUEST['filterByStatus'])) {
            $this->filterByStatus = StringUtil::trim($_REQUEST['filterByStatus']);
        }

        // language filter
        if (isset($_REQUEST['languageID'])) {
            $this->languageID = \intval($_REQUEST['languageID']);
        }

        // read available label groups
        $labelGroupIDs = BoardCache::getInstance()->getLabelGroupIDs($this->boardID);
        if (!empty($labelGroupIDs)) {
            $this->labelGroups = LabelHandler::getInstance()->getLabelGroups($labelGroupIDs, true, 'canViewLabel');

            if (isset($_REQUEST['labelIDs']) && \is_array($_REQUEST['labelIDs'])) {
                $this->labelIDs = $_REQUEST['labelIDs'];

                foreach ($this->labelIDs as $groupID => $labelID) {
                    $isValid = false;

                    // ignore zero-values
                    if (!\is_array($labelID) && $labelID) {
                        if (
                            isset($this->labelGroups[$groupID])
                            && ($labelID == -1 || $this->labelGroups[$groupID]->isValid($labelID))
                        ) {
                            $isValid = true;
                        }
                    }

                    if (!$isValid) {
                        unset($this->labelIDs[$groupID]);
                    }
                }
            }
        }

        // build link parameters
        $tagLinkParameters = [
            'sortField' => $this->sortField,
            'sortOrder' => $this->sortOrder,
            'filterByDate' => $this->filterByDate,
            'filterByStatus' => $this->filterByStatus,
            'pageNo' => $this->pageNo,
        ];
        if (!empty($this->languageID)) {
            $tagLinkParameters['languageID'] = $this->languageID;
        }
        if (!empty($this->labelIDs)) {
            $tagLinkParameters['labelIDs'] = [];
            foreach ($this->labelIDs as $groupID => $labelID) {
                $tagLinkParameters['labelIDs'][$groupID] = $labelID;
            }
        }
        $this->tagLinkParameters = \http_build_query($tagLinkParameters, '', '&');

        // read available tags
        if (
            MODULE_TAGGING
            && WCF::getSession()->getPermission('user.tag.canViewTag')
            && (!$this->board->isPrivate || $this->board->canReadPrivateThreads())
        ) {
            $tagCloud = new BoardTagCloud($this->board->boardID);
            $tags = $tagCloud->getTags();

            if (!empty($tags) && isset($_REQUEST['tagID'])) {
                $tagID = \intval($_REQUEST['tagID']);
                foreach ($tags as $tag) {
                    if ($tag->tagID == $tagID) {
                        $this->tagID = $tagID;
                        break;
                    }
                }
            }
        }

        if (empty($_POST)) {
            $this->canonicalURL = LinkHandler::getInstance()->getLink('Board', [
                'application' => 'wbb',
                'object' => $this->board,
            ], ($this->pageNo > 1 ? 'pageNo=' . $this->pageNo : ''));
        } else {
            HeaderUtil::redirect(LinkHandler::getInstance()->getLink('Board', [
                'application' => 'wbb',
                'object' => $this->board,
            ], $this->tagLinkParameters));

            exit;
        }
    }

    /**
     * @inheritDoc
     */
    public function countItems()
    {
        if (!$this->board->isBoard()) {
            return 0;
        }

        return parent::countItems();
    }

    /**
     * @inheritDoc
     */
    public function calculateNumberOfPages()
    {
        parent::calculateNumberOfPages();

        // check for announcements since they're excluded within countObjects()
        if ($this->board->isBoard() && $this->items == 0) {
            $this->items = $this->objectList->countAnnouncements();
        }
    }

    /**
     * @inheritDoc
     */
    protected function initObjectList()
    {
        if ($this->board->isBoard()) {
            $this->objectList = new BoardThreadList(
                $this->board,
                $this->filterByDate,
                $this->labelIDs,
                $this->tagID,
                $this->filterByStatus,
                $this->languageID
            );

            $ignoredUsers = UserProfileHandler::getInstance()->getIgnoredUsers(UserIgnore::TYPE_HIDE_MESSAGES);
            if (!empty($ignoredUsers)) {
                $this->objectList->getConditionBuilder()->add(
                    "(thread.userID IS NULL OR thread.userID NOT IN (?))",
                    [$ignoredUsers]
                );
            }

            if ($this->filterByDate != WBB_BOARD_DEFAULT_DAYS_PRUNE) {
                $this->hasActiveFilter = true;
            } elseif (!empty($this->labelIDs)) {
                $this->hasActiveFilter = true;
            } elseif ($this->filterByStatus) {
                $this->hasActiveFilter = true;
            } elseif ($this->languageID) {
                $this->hasActiveFilter = true;
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        if (!MODULE_LIKE) {
            $this->validSortFields = \array_filter($this->validSortFields, static function ($sortField) {
                return $sortField !== 'cumulativeLikes';
            });
        }

        parent::readData();

        // get node tree
        $this->boardNodeList = new DetailedBoardNodeList($this->board->boardID);
        $this->boardNodeList->readNodeTree();

        // add breadcrumbs
        WBBCore::getInstance()->setLocation($this->board->getParentBoards());

        // read available content languages
        $this->availableLanguages = LanguageFactory::getInstance()->getContentLanguages();

        // assigned moderators
        if (WBB_BOARD_ENABLE_MODERATORS) {
            $userIDs = BoardCache::getInstance()->getUserModerators($this->board->boardID);
            if (!empty($userIDs)) {
                $this->userModerators = \array_filter(UserProfileRuntimeCache::getInstance()->getObjects($userIDs));
            }
            $groupIDs = BoardCache::getInstance()->getGroupModerators($this->board->boardID);
            if (!empty($groupIDs)) {
                foreach ($groupIDs as $groupID) {
                    if (($group = UserGroup::getGroupByID($groupID)) !== null) {
                        $this->groupModerators[] = $group;
                    }
                }
            }
        }

        // set meta description
        if ($this->board->description || $this->board->metaDescription) {
            if ($this->board->metaDescription) {
                $description = WCF::getLanguage()->get($this->board->metaDescription);
            } else {
                $description = ($this->board->descriptionUseHtml ? StringUtil::decodeHTML(StringUtil::stripHTML(WCF::getLanguage()->get($this->board->description))) : WCF::getLanguage()->get($this->board->description));
            }

            MetaTagHandler::getInstance()->addTag('description', 'description', $description);
        }
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        $hasMarkedItems = ClipboardHandler::getInstance()->hasMarkedItems(
            ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.thread')
        );
        if (!$hasMarkedItems) {
            // check if there are any posts marked
            $hasMarkedItems = ClipboardHandler::getInstance()->hasMarkedItems(
                ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.post')
            );
        }

        WCF::getTPL()->assign([
            'availableLanguages' => $this->availableLanguages,
            'boardID' => $this->boardID,
            'board' => $this->board,
            'boardNodeList' => $this->boardNodeList->getNodeList(),
            'filterByDate' => $this->filterByDate,
            'filterByStatus' => $this->filterByStatus,
            'hasMarkedItems' => $hasMarkedItems,
            'labelGroups' => $this->labelGroups,
            'labelIDs' => $this->labelIDs,
            'userModerators' => $this->userModerators,
            'groupModerators' => $this->groupModerators,
            'languageID' => $this->languageID,
            'tagID' => $this->tagID,
            'defaultFilterByDate' => $this->board->daysPrune ?: WBB_BOARD_DEFAULT_DAYS_PRUNE,
            'defaultSortField' => $this->defaultSortField,
            'defaultSortOrder' => $this->defaultSortOrder,
            'validSortFields' => $this->validSortFields,
            'hasActiveFilter' => $this->hasActiveFilter,
        ]);

        if (
            $this->board->isBoard()
            && \count($this->objectList) === 0
            && \count(BoardCache::getInstance()->getChildIDs($this->board->boardID)) === 0
        ) {
            @\header('HTTP/1.1 404 Not Found');
        }
    }

    /**
     * @inheritDoc
     */
    public function getBoard()
    {
        return $this->board;
    }
}
