<?php

namespace wbb\system\log\modification;

use wbb\data\board\Board;
use wbb\data\modification\log\ViewableThreadModificationLog;
use wbb\data\post\Post;
use wbb\data\thread\Thread;
use wbb\system\cache\runtime\ThreadRuntimeCache;
use wbb\system\user\notification\object\ThreadModerationUserNotificationObject;
use wcf\data\label\Label;
use wcf\data\modification\log\ModificationLog;
use wcf\system\cache\runtime\UserRuntimeCache;
use wcf\system\label\LabelHandler;
use wcf\system\log\modification\AbstractExtendedModificationLogHandler;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\WCF;

/**
 * Handles thread modification logs.
 *
 * @author      Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license     WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package     WoltLabSuite\Forum\System\Log\Modification
 */
class ThreadModificationLogHandler extends AbstractExtendedModificationLogHandler
{
    /**
     * @inheritDoc
     */
    protected $objectTypeName = 'com.woltlab.wbb.thread';

    /**
     * Adds a log entry for thread close.
     *
     * @param        Thread $thread
     */
    public function close(Thread $thread)
    {
        $this->fireNotification($thread, $this->add($thread, 'close', [], 0));
    }

    /**
     * Adds a log entry for thread delete.
     *
     * @param        Thread $thread
     */
    public function delete(Thread $thread)
    {
        $this->add($thread, 'delete', [
            'time' => $thread->time,
            'topic' => $thread->topic,
        ]);
    }

    /**
     * Adds a log entry for thread marking as done.
     *
     * @param        Thread $thread
     */
    public function done(Thread $thread)
    {
        $this->add($thread, 'done');
    }

    /**
     * Adds a log entry for thread enable.
     *
     * @param        Thread $thread
     */
    public function enable(Thread $thread)
    {
        $this->fireNotification($thread, $this->add($thread, 'enable', [], 0));
    }

    /**
     * Adds a log entry for thread disable.
     *
     * @param        Thread $thread
     */
    public function disable(Thread $thread)
    {
        $this->add($thread, 'disable');
    }

    /**
     * Adds a log entry for thread move.
     *
     * @param        Thread $thread
     * @param        Board $oldBoard
     * @param        Board $newBoard
     */
    public function move(Thread $thread, Board $oldBoard, Board $newBoard)
    {
        $this->fireNotification($thread, $this->add($thread, 'move', [
            'oldBoard' => [
                'boardID' => $oldBoard->boardID,
                'title' => $oldBoard->title,
            ],
            'newBoard' => [
                'boardID' => $newBoard->boardID,
                'title' => $newBoard->title,
            ],
        ], 0));
    }

    /**
     * Adds a log entry for thread open.
     *
     * @param        Thread $thread
     */
    public function open(Thread $thread)
    {
        $this->fireNotification($thread, $this->add($thread, 'open', [], 0));
    }

    /**
     * Adds a log entry for thread restore.
     *
     * @param        Thread $thread
     */
    public function restore(Thread $thread)
    {
        $this->fireNotification($thread, $this->add($thread, 'restore', [], 0));
    }

    /**
     * Adds a log entry for thread soft-delete (trash).
     *
     * @param        Thread $thread
     * @param        string $reason
     */
    public function trash(Thread $thread, $reason = '')
    {
        $this->fireNotification($thread, $this->add($thread, 'trash', ['reason' => $reason], 0));
    }

    /**
     * Adds a log entry for thread marking as undone.
     *
     * @param        Thread $thread
     */
    public function undone(Thread $thread)
    {
        $this->add($thread, 'undone');
    }

    /**
     * Adds a log entry for changed labels.
     *
     * @param        Thread $thread
     * @param        Label $label
     * @param        Label $oldLabel
     */
    public function setLabel(Thread $thread, ?Label $label = null, ?Label $oldLabel = null)
    {
        $this->fireNotification($thread, $this->add($thread, 'setLabel', [
            'label' => $label,
            'oldLabel' => $oldLabel,
        ], 0));
    }

    /**
     * Adds a log entry for pinning a thread.
     *
     * @param        Thread $thread
     * @since        5.0
     */
    public function sticky(Thread $thread)
    {
        $this->add($thread, 'sticky');
    }

    /**
     * Adds a log entry for unpinning a thread.
     *
     * @param        Thread $thread
     * @since        5.0
     */
    public function scrape(Thread $thread)
    {
        $this->add($thread, 'scrape');
    }

    /**
     * Adds a log entry for changing the thread's topic.
     *
     * @param        Thread $thread
     * @param        string $newTopic
     * @since        5.0
     */
    public function changeTopic(Thread $thread, $newTopic)
    {
        $this->fireNotification($thread, $this->add($thread, 'changeTopic', [
            'oldTopic' => $thread->topic,
            'newTopic' => $newTopic,
        ], 0));
    }

    /**
     * Adds a log entry for merging threads.
     *
     * @param        Thread $thread
     * @param        Thread[] $mergedThreads
     * @since        5.0
     */
    public function merge(Thread $thread, array $mergedThreads)
    {
        $threadTitles = [];
        foreach ($mergedThreads as $mergedThread) {
            $threadTitles[] = $mergedThread->getTitle();
        }

        $this->add($thread, 'merge', ['mergedThreads' => $threadTitles]);
    }

    /**
     * Adds a log entry for marking a post as the best answer.
     *
     * @since        5.4
     */
    public function markAsBestAnswer(Thread $thread, Post $post): void
    {
        $this->add($thread, 'markAsBestAnswer', ['postID' => $post->postID], 0);
    }

    /**
     * Adds a log entry for setting the thread as an announcement.
     *
     * @param        Thread $thread
     * @since        5.0
     */
    public function setAsAnnouncement(Thread $thread)
    {
        $this->add($thread, 'setAsAnnouncement');
    }

    /**
     * Adds a log entry for unsetting the thread as an announcement.
     *
     * @param        Thread $thread
     * @since        5.0
     */
    public function unsetAsAnnouncement(Thread $thread)
    {
        $this->add($thread, 'unsetAsAnnouncement');
    }

    /**
     * Adds a thread modification log entry.
     *
     * @param        Thread $thread
     * @param        string $action
     * @param        array $additionalData
     * @param        int $hidden
     * @return        ModificationLog
     */
    public function add(Thread $thread, $action, array $additionalData = [], $hidden = 1)
    {
        return $this->createLog(
            $action,
            $thread->threadID,
            $thread->boardID,
            $additionalData,
            TIME_NOW,
            null,
            null,
            $hidden
        );
    }

    /**
     * Fires a moderation notification.
     *
     * @param        Thread $thread
     * @param        ModificationLog $modificationLog
     * @since        5.1
     */
    protected function fireNotification(Thread $thread, ModificationLog $modificationLog)
    {
        if (!WBB_THREAD_ENABLE_MODERATION_NOTIFICATION) {
            return;
        }

        if ($thread->userID == WCF::getUser()->userID || !$thread->userID) {
            return;
        }

        if (!$thread->canRead() && !($thread->getBoard()->getPermission() && $modificationLog->action == 'trash')) {
            return;
        }

        // validate label permissions
        if ($modificationLog->action == 'setLabel' && $thread->getBoard()->getPermission()) {
            // check permissions
            if ($modificationLog->label instanceof Label) {
                $labelID = $modificationLog->label->labelID;
            } else {
                if ($modificationLog->oldLabel instanceof Label) {
                    $labelID = $modificationLog->oldLabel->labelID;
                } else {
                    throw new \LogicException('Unreachable');
                }
            }

            if (!LabelHandler::getInstance()->validateCanView([$labelID], UserRuntimeCache::getInstance()->getObject($thread->userID))[$labelID]) {
                return;
            }
        }

        UserNotificationHandler::getInstance()->fireEvent(
            'moderate',
            'com.woltlab.wbb.moderation.thread',
            new ThreadModerationUserNotificationObject($modificationLog),
            [
                $thread->userID,
            ]
        );
    }

    /**
     * @inheritDoc
     */
    public function getAvailableActions()
    {
        return [
            'changeTopic',
            'close',
            'delete',
            'disable',
            'done',
            'enable',
            'markAsBestAnswer',
            'merge',
            'move',
            'open',
            'restore',
            'scrape',
            'setAsAnnouncement',
            'setLabel',
            'sticky',
            'trash',
            'undone',
            'unsetAsAnnouncement',
        ];
    }

    /**
     * @inheritDoc
     */
    public function processItems(array $items)
    {
        $threadIDs = [];
        /** @var ModificationLog $item */
        foreach ($items as &$item) {
            $threadIDs[] = $item->objectID;

            $item = new ViewableThreadModificationLog($item);
        }
        unset($item);

        if (!empty($threadIDs)) {
            $threads = ThreadRuntimeCache::getInstance()->getObjects($threadIDs);
            /** @var ViewableThreadModificationLog $item */
            foreach ($items as $item) {
                if (isset($threads[$item->objectID])) {
                    $item->setThread($threads[$item->objectID]);
                }
            }
        }

        return $items;
    }
}
