<?php

namespace wbb\data\post;

use wbb\data\modification\log\ThreadModificationLogList;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadList;
use wcf\data\attachment\GroupedAttachmentList;
use wcf\data\moderation\queue\ModerationQueue;
use wcf\data\object\type\ObjectTypeCache;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\message\embedded\object\MessageEmbeddedObjectManager;
use wcf\system\poll\PollManager;
use wcf\system\WCF;

/**
 * Represents a list of viewable posts.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Posts
 *
 * @method  ViewablePost        current()
 * @method  ViewablePost[]      getObjects()
 * @method  ViewablePost|null   getSingleObject()
 * @method  ViewablePost|null   search($objectID)
 * @property    ViewablePost[]      $objects
 */
class ViewablePostList extends PostList
{
    /**
     * @inheritDoc
     */
    public $sqlOrderBy = 'post.time, post.postID';

    /**
     * @inheritDoc
     */
    public $decoratorClassName = ViewablePost::class;

    /**
     * attachment object ids
     * @var int[]
     */
    public $attachmentObjectIDs = [];

    /**
     * poll ids
     * @var int[]
     */
    public $pollIDs = [];

    /**
     * ids of the posts with embedded objects
     * @var int[]
     */
    public $embeddedObjectPostIDs = [];

    /**
     * list of the associated thread ids
     * @var int[]
     */
    public $threadIDs = [];

    /**
     * attachment list
     * @var GroupedAttachmentList
     */
    protected $attachmentList;

    /**
     * thread modification log list
     * @var ThreadModificationLogList
     */
    public $logList;

    /**
     * max post time
     * @var int
     */
    protected $maxPostTime = 0;

    /**
     * thread object
     * @var Thread
     */
    public $thread;

    /**
     * enables/disables the loading of attachments
     * @var bool
     */
    protected $attachmentLoading = false;

    /**
     * enables/disables the loading of polls
     * @var bool
     */
    protected $pollLoading = true;

    /**
     * enables/disables the loading of modification log entries
     * @var bool
     */
    protected $modificationLogLoading = true;

    /**
     * enables/disables the loading of embedded objects
     * @var bool
     */
    protected $embeddedObjectLoading = true;

    /**
     * enables/disables the loading of threads
     * @var bool
     */
    protected $threadLoading = false;

    /**
     * @inheritDoc
     */
    public function __construct()
    {
        parent::__construct();

        // get report status
        if (WCF::getSession()->getPermission('mod.general.canUseModeration')) {
            if (!empty($this->sqlSelects)) {
                $this->sqlSelects .= ',';
            }
            $this->sqlSelects .= "moderation_queue.queueID AS reportQueueID";
            $this->sqlJoins .= "
                LEFT JOIN   wcf" . WCF_N . "_moderation_queue moderation_queue
                ON          moderation_queue.objectTypeID = " . ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.moderation.report', 'com.woltlab.wbb.post') . "
                        AND moderation_queue.objectID = post.postID
                        AND moderation_queue.status <> " . ModerationQueue::STATUS_DONE . "
                        AND moderation_queue.status <> " . ModerationQueue::STATUS_REJECTED . "
                        AND moderation_queue.status <> " . ModerationQueue::STATUS_CONFIRMED;
        }
    }

    /**
     * @inheritDoc
     */
    public function readObjects()
    {
        if ($this->objectIDs === null) {
            $this->readObjectIDs();
        }
        parent::readObjects();

        $postIDs = $userIDs = [];
        foreach ($this->objects as $post) {
            if ($post->time > $this->maxPostTime) {
                $this->maxPostTime = $post->time;
            }
            if ($this->thread !== null) {
                $post->setThread($this->thread);
            }

            if ($post->attachments) {
                $this->attachmentObjectIDs[] = $post->postID;
            }
            if ($post->pollID) {
                $this->pollIDs[] = $post->pollID;
            }
            if ($post->hasEmbeddedObjects) {
                $this->embeddedObjectPostIDs[] = $post->postID;
            }

            if (($this->thread !== null && !$this->thread->isDeleted && $post->isDeleted) || $this->thread === null) {
                $postIDs[] = $post->postID;
            }

            if ($post->userID) {
                $userIDs[] = $post->userID;
            }

            $this->threadIDs[] = $post->threadID;
        }

        // cache user ids
        if (!empty($userIDs)) {
            UserProfileRuntimeCache::getInstance()->cacheObjectIDs(\array_unique($userIDs));
        }

        if ($this->modificationLogLoading && !empty($postIDs)) {
            $this->logList = new ThreadModificationLogList();
            $this->logList->setPostData($postIDs, 'trash');
            $this->logList->readObjects();

            foreach ($this->logList as $logEntry) {
                foreach ($this->objects as $post) {
                    if ($post->postID == $logEntry->objectID) {
                        $post->setLogEntry($logEntry);
                    }
                }
            }
        }

        if ($this->embeddedObjectLoading) {
            $this->readEmbeddedObjects();
        }
        if ($this->attachmentLoading) {
            $this->readAttachments();
        }
        if ($this->pollLoading) {
            $this->readPolls();
        }
        if ($this->threadLoading) {
            $this->readThreads();
        }
    }

    /**
     * Reads the embedded objects of the posts in the list.
     */
    public function readEmbeddedObjects()
    {
        if (!empty($this->embeddedObjectPostIDs)) {
            // add post objects to attachment object cache to save SQL queries
            ObjectTypeCache::getInstance()
                ->getObjectTypeByName('com.woltlab.wcf.attachment.objectType', 'com.woltlab.wbb.post')
                ->getProcessor()
                ->setCachedObjects($this->objects);

            // load embedded objects
            MessageEmbeddedObjectManager::getInstance()->loadObjects(
                'com.woltlab.wbb.post',
                $this->embeddedObjectPostIDs
            );
        }
    }

    /**
     * Reads the attachments of the posts in the list.
     */
    public function readAttachments()
    {
        if (!empty($this->attachmentObjectIDs)) {
            $this->attachmentList = new GroupedAttachmentList('com.woltlab.wbb.post');
            $this->attachmentList->getConditionBuilder()->add(
                'attachment.objectID IN (?)',
                [$this->attachmentObjectIDs]
            );
            $this->attachmentList->readObjects();
        }
    }

    /**
     * Reads the polls of the posts in the list.
     */
    public function readPolls()
    {
        if (MODULE_POLL && !empty($this->pollIDs)) {
            $polls = PollManager::getInstance()->getPolls($this->pollIDs);

            foreach ($polls as $poll) {
                if (isset($this->objects[$poll->objectID])) {
                    $this->objects[$poll->objectID]->setPoll($poll);
                }
            }
        }
    }

    /**
     * Reads the threads for the posts in the list.
     */
    public function readThreads()
    {
        if (!empty($this->threadIDs)) {
            $threadList = new ThreadList();
            $threadList->setObjectIDs($this->threadIDs);
            $threadList->readObjects();
            $threads = $threadList->getObjects();

            foreach ($this->objects as $post) {
                $post->setThread($threads[$post->threadID]);
            }
        }
    }

    /**
     * Returns the max post time
     *
     * @return  int
     */
    public function getMaxPostTime()
    {
        return $this->maxPostTime;
    }

    /**
     * Returns the list of attachments
     *
     * @return  GroupedAttachmentList
     */
    public function getAttachmentList()
    {
        if ($this->attachmentList === null) {
            $this->readAttachments();
        }

        return $this->attachmentList;
    }

    /**
     * Sets active thread.
     *
     * @param   Thread      $thread
     */
    public function setThread(Thread $thread)
    {
        $this->thread = $thread;
    }

    /**
     * Returns thread modification log list.
     *
     * @return  ThreadModificationLogList
     */
    public function getLogList()
    {
        return $this->logList;
    }

    /**
     * Enables/disables the loading of attachments.
     *
     * @param   bool        $enable
     */
    public function enableAttachmentLoading($enable = true)
    {
        $this->attachmentLoading = $enable;
    }

    /**
     * Enables/disables the loading of polls.
     *
     * @param   bool        $enable
     */
    public function enablePollLoading($enable = true)
    {
        $this->pollLoading = $enable;
    }

    /**
     * Enables/disables the loading of embedded objects.
     *
     * @param   bool        $enable
     */
    public function enableEmbeddedObjectLoading($enable = true)
    {
        $this->embeddedObjectLoading = $enable;
    }

    /**
     * Enables/disables the loading of modification log entries.
     *
     * @param   bool        $enable
     */
    public function enableModificationLogLoading($enable = true)
    {
        $this->modificationLogLoading = $enable;
    }

    /**
     * Enables/disables the loading of threads.
     *
     * @param       bool         $enable
     */
    public function enableThreadLoading($enable = true)
    {
        $this->threadLoading = $enable;
    }
}
