<?php

namespace wbb\system\user\notification\event;

use wbb\system\cache\runtime\ThreadRuntimeCache;
use wbb\system\user\notification\object\ThreadModerationUserNotificationObject;
use wcf\data\label\Label;
use wcf\data\user\UserProfile;
use wcf\system\label\LabelHandler;
use wcf\system\user\notification\event\AbstractSharedUserNotificationEvent;
use wcf\system\user\notification\event\ITestableUserNotificationEvent;
use wcf\system\user\notification\event\TTestableUserNotificationEvent;

/**
 * User notification event for moderating threads.
 *
 * @author  Joshua Ruesweg
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\User\Notification\Event
 * @since   5.1
 *
 * @method  ThreadModerationUserNotificationObject  getUserNotificationObject()
 */
class ModerateThreadUserNotificationEvent extends AbstractSharedUserNotificationEvent implements
    ITestableUserNotificationEvent
{
    use TTestablePostRelatedUserNotificationEvent;
    use TTestableUserNotificationEvent;

    /**
     * @inheritDoc
     */
    protected function prepare()
    {
        ThreadRuntimeCache::getInstance()->cacheObjectID($this->getUserNotificationObject()->objectID);
    }

    /**
     * @inheritDoc
     */
    public function getTitle()
    {
        return $this->getLanguage()->getDynamicVariable(
            'wbb.thread.notification.moderate.' . $this->getUserNotificationObject()->action . '.notification.title',
            [
                'userNotificationObject' => $this->getUserNotificationObject(),
            ]
        );
    }

    /**
     * @inheritDoc
     */
    public function getMessage()
    {
        return $this->getLanguage()->getDynamicVariable(
            'wbb.thread.notification.moderate.' . $this->getUserNotificationObject()->action . '.notification.message',
            [
                'author' => $this->author,
                'thread' => $this->getUserNotificationObject()->getThread(),
                'userNotificationObject' => $this->getUserNotificationObject(),
            ]
        );
    }

    /**
     * @inheritDoc
     */
    public function getLink()
    {
        return $this->getUserNotificationObject()->getURL();
    }

    /**
     * @inheritDoc
     */
    public function getEventHash()
    {
        return \sha1($this->eventID . '-' . $this->getUserNotificationObject()->getObjectID());
    }

    /**
     * @inheritDoc
     */
    public function supportsEmailNotification()
    {
        return false;
    }

    /**
     * @inheritDoc
     */
    public function checkAccess()
    {
        if (!WBB_THREAD_ENABLE_MODERATION_NOTIFICATION) {
            return false;
        }

        if (!$this->getUserNotificationObject()->getThread()) {
            return false;
        }

        if (
            !$this->getUserNotificationObject()->getThread()->canRead()
            && !(
                $this->getUserNotificationObject()->getThread()->getBoard()->getPermission()
                && $this->getUserNotificationObject()->action == 'trash'
            )
        ) {
            return false;
        }

        // validate label permissions
        if (
            $this->getUserNotificationObject()->action == 'setLabel'
            && $this->getUserNotificationObject()->getThread()->getBoard()->getPermission()
        ) {
            // check permissions
            if ($this->getUserNotificationObject()->label instanceof Label) {
                $labelID = $this->getUserNotificationObject()->label->labelID;
            } elseif ($this->getUserNotificationObject()->oldLabel instanceof Label) {
                $labelID = $this->getUserNotificationObject()->oldLabel->labelID;
            } else {
                throw new \LogicException('Unreachable');
            }

            if (!LabelHandler::getInstance()->validateCanView([$labelID])[$labelID]) {
                return false;
            }
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public static function canBeTriggeredByGuests()
    {
        return true;
    }

    /**
     * @inheritDoc
     */
    public static function getTestObjects(UserProfile $recipient, UserProfile $author)
    {
        $newLabel = new Label(null, [
            'labelID' => 10000000000,
            'groupID' => 10000000000,
            'label' => 'New Thread Test Label',
            'cssClassName' => 'green',
            'showOrder' => 1,
        ]);
        $oldLabel = new Label(null, [
            'labelID' => 10000000001,
            'groupID' => 10000000000,
            'label' => 'Old Thread Test Label',
            'cssClassName' => 'red',
            'showOrder' => 1,
        ]);

        return [
            new ThreadModerationUserNotificationObject(
                self::createTestThreadModificationLog($recipient, $author, 'close')
            ),
            new ThreadModerationUserNotificationObject(
                self::createTestThreadModificationLog($recipient, $author, 'enable')
            ),
            new ThreadModerationUserNotificationObject(
                self::createTestThreadModificationLog($recipient, $author, 'move')
            ),
            new ThreadModerationUserNotificationObject(
                self::createTestThreadModificationLog($recipient, $author, 'setLabel', [
                    'label' => $newLabel,
                ])
            ),
            new ThreadModerationUserNotificationObject(
                self::createTestThreadModificationLog($recipient, $author, 'setLabel', [
                    'label' => $newLabel,
                    'oldLabel' => $oldLabel,
                ])
            ),
            new ThreadModerationUserNotificationObject(
                self::createTestThreadModificationLog($recipient, $author, 'setLabel', [
                    'oldLabel' => $oldLabel,
                ])
            ),
            new ThreadModerationUserNotificationObject(
                self::createTestThreadModificationLog($recipient, $author, 'trash', [
                    'reason' => 'Test Reason',
                ])
            ),
        ];
    }
}
