<?php

namespace wbb\system\user\activity\event;

use wbb\data\post\SimplifiedViewablePostList;
use wbb\data\thread\ViewableThreadList;
use wcf\system\SingletonFactory;
use wcf\system\user\activity\event\IUserActivityEvent;
use wcf\system\WCF;

/**
 * User activity event implementation for liked posts.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\User\Activity\Event
 */
class LikeablePostUserActivityEvent extends SingletonFactory implements IUserActivityEvent
{
    /**
     * @inheritDoc
     */
    public function prepare(array $events)
    {
        if (!WCF::getSession()->getPermission('user.like.canViewLike')) {
            return;
        }

        $postIDs = [];
        foreach ($events as $event) {
            $postIDs[] = $event->objectID;
        }

        // fetch posts
        $postList = new SimplifiedViewablePostList();
        $postList->setObjectIDs($postIDs);
        $postList->readObjects();
        $posts = $postList->getObjects();

        // fetch threads
        $threadIDs = $threads = [];
        foreach ($posts as $post) {
            $threadIDs[] = $post->threadID;
        }
        if (!empty($threadIDs)) {
            $threadList = new ViewableThreadList();
            $threadList->setObjectIDs($threadIDs);
            $threadList->readObjects();
            $threads = $threadList->getObjects();
        }

        // set message
        foreach ($events as $event) {
            if (isset($posts[$event->objectID])) {
                $post = $posts[$event->objectID];
                $thread = $threads[$post->threadID];
                $post->setThread($thread->getDecoratedObject());

                // check permissions
                if (!$post->canRead()) {
                    continue;
                }
                // check ignored boards
                if (WCF::getUser()->userID && $thread->getBoard()->isIgnored()) {
                    continue;
                }
                // check ignored thread
                if (WCF::getUser()->userID && $thread->isIgnored()) {
                    continue;
                }
                $event->setIsAccessible();

                // short output
                $text = WCF::getLanguage()->getDynamicVariable('wcf.user.profile.recentActivity.likedPost', [
                    'post' => $post,
                    'thread' => $threads[$post->threadID],
                    'reactionType' => $event->reactionType,
                ]);
                $event->setTitle($text);

                // output
                $event->setDescription($post->getExcerpt());
            } else {
                $event->setIsOrphaned();
            }
        }
    }
}
