<?php

namespace wbb\system\page\handler;

use wbb\data\board\Board;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\WCF;

/**
 * Default implementation for page handlers that are based on the thread status (subscribe / ignore).
 *
 * @author  Marcel Werk
 * @copyright   2001-2022 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Page\Handler
 * @since   5.5
 */
trait TThreadStatusPageHandler
{
    /**
     * Returns true if the active has threads with the given status.
     */
    private function hasStatusThreads(string $status, string $storageFieldName): bool
    {
        if (!WCF::getUser()->userID) {
            return false;
        }

        $count = 0;
        $data = UserStorageHandler::getInstance()->getField($storageFieldName);

        // cache does not exist or is outdated
        if ($data === null) {
            $boardIDs = Board::filterBoardIDs(Board::getAccessibleBoardIDs());
            $privateBoardIDs = Board::filterBoardIDs(Board::getPrivateBoardIDs());
            if (!empty($boardIDs) || !empty($privateBoardIDs)) {
                $conditionBuilder = new PreparedStatementConditionBuilder();

                if (empty($privateBoardIDs)) {
                    $conditionBuilder->add("thread.boardID IN (?)", [$boardIDs]);
                } elseif (empty($boardIDs)) {
                    $conditionBuilder->add(
                        "(thread.boardID IN (?) AND thread.userID = ?)",
                        [$privateBoardIDs, WCF::getUser()->userID]
                    );
                } else {
                    $conditionBuilder->add(
                        "(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))",
                        [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]
                    );
                }

                $conditionBuilder->add(
                    'thread_user_status.status = ?',
                    [$status]
                );
                $conditionBuilder->add(
                    'thread_user_status.userID = ?',
                    [WCF::getUser()->userID]
                );
                $conditionBuilder->add('thread.isDeleted = 0 AND thread.isDisabled = 0');

                $sql = "SELECT      COUNT(*) AS count
                        FROM        wbb1_thread_user_status thread_user_status
                        LEFT JOIN   wbb1_thread thread
                        ON          thread.threadID = thread_user_status.threadID
                        " . $conditionBuilder;
                $statement = WCF::getDB()->prepare($sql);
                $statement->execute($conditionBuilder->getParameters());
                $count = $statement->fetchSingleColumn();
            }

            // update storage data
            UserStorageHandler::getInstance()->update(WCF::getUser()->userID, $storageFieldName, $count);
        } else {
            $count = $data;
        }

        return $count != 0;
    }
}
