/**
 * Basic update handler for threads.
 *
 * @author  Joshua Ruesweg
 * @copyright  2001-2021 WoltLab GmbH
 * @license  WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module  WoltLabSuite/Forum/Handler/Thread/UpdateHandler
 * @since 5.5
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Controller/Clipboard", "WoltLabSuite/Core/Event/Handler"], function (require, exports, tslib_1, Clipboard, EventHandler) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.UpdateHandler = void 0;
    Clipboard = tslib_1.__importStar(Clipboard);
    EventHandler = tslib_1.__importStar(EventHandler);
    class UpdateHandler {
        constructor() {
            this.threads = new Map();
            document.querySelectorAll(".wbbThread").forEach((element) => {
                let threadId;
                if (!element.dataset.threadId) {
                    throw new Error("The threadID is not set for the `.wbbThread` element.");
                }
                else {
                    threadId = parseInt(element.dataset.threadId, 10);
                }
                if (this.threads.get(threadId)) {
                    throw new Error(`The element with the threadId "${threadId}" is read multiple times.`);
                }
                this.threads.set(threadId, element);
            });
        }
        /** @deprecated */
        setMarkAsDoneHandler() {
            // Does nothing, is deprecated.
        }
        update(threadId, data, reloadClipboard) {
            if ("moved" in data && data.moved) {
                window.location.reload();
                return;
            }
            // Workaround for old JS, which gives the threadId as string.
            // Remove this workaround in a future version!
            threadId = ~~threadId;
            if (!this.getThread(threadId)) {
                throw new Error(`Unknown thread with the threadId "${threadId}".`);
            }
            for (const property in data) {
                this.updateProperty(threadId, property, data[property]);
            }
            if (reloadClipboard) {
                Clipboard.reload();
            }
        }
        updateProperty(threadId, property, value) {
            if (/^is[a-zA-Z]+$/g.test(property)) {
                this.getThread(threadId).dataset[property] = value;
            }
            switch (property) {
                case "deleted":
                    this.delete(threadId, value);
                    break;
                case "isClosed":
                    if (value) {
                        this.close(threadId);
                    }
                    else {
                        this.open(threadId);
                    }
                    break;
                case "isDeleted":
                    if (value) {
                        this.trash(threadId);
                    }
                    else {
                        this.restore(threadId);
                    }
                    break;
                case "isDisabled":
                    if (value) {
                        this.disable(threadId);
                    }
                    else {
                        this.enable(threadId);
                    }
                    break;
                case "isDone":
                    if (value) {
                        this.done(threadId);
                    }
                    else {
                        this.undone(threadId);
                    }
                    break;
                case "isSticky":
                    if (value) {
                        this.sticky(threadId);
                    }
                    else {
                        this.scrape(threadId);
                    }
                    break;
                case "moved":
                    this.moved(threadId);
                    break;
                case "showMoveNotice":
                    this.showMoveNotice(threadId);
                    break;
                case "title":
                    this.updateTitle(threadId, value);
                    break;
                default:
                    this.handleCustomEvent(threadId, property, value);
            }
        }
        handleCustomEvent(threadId, property, value) {
            /** @deprecated: Will be removed with WoltLab Suite Forum 5.6 */
            window.jQuery(this.getThread(threadId)).trigger("threadUpdateHandlerProperty", [threadId, property, value]);
            EventHandler.fire("com.woltlab.wbb.updateHandler", property, {
                threadId,
                value,
                thread: this.getThread(threadId),
            });
        }
        getValue(threadId, property) {
            // Workaround for old JS, which gives the threadId as string.
            // Remove this workaround in a future version!
            threadId = ~~threadId;
            switch (property) {
                case "isAnnouncement":
                case "isClosed":
                case "isDeleted":
                case "isDisabled":
                case "isDone":
                case "isSticky":
                case "isLink":
                case "canMarkAsDone":
                    return this.getThread(threadId).dataset[property] === "1";
                default:
                    throw new Error(`Unknown property "${property}".`);
            }
        }
        getThread(threadId) {
            if (this.threads.get(threadId)) {
                return this.threads.get(threadId);
            }
            throw new Error(`Unknown thread with the threadId "${threadId}".`);
        }
    }
    exports.UpdateHandler = UpdateHandler;
    exports.default = UpdateHandler;
});
