<?php

namespace wbb\system\thread\editor;

use wbb\data\board\BoardCache;
use wbb\data\board\RealtimeBoardNodeList;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadAction;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\language\LanguageFactory;
use wcf\system\message\censorship\Censorship;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Provides an editor for thread properties.
 *
 * @author  Alexander Ebert
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\System\Thread\Editor
 */
class DefaultThreadEditor extends AbstractThreadEditor
{
    /**
     * @inheritDoc
     */
    public function beginEdit()
    {
        $type = Thread::TYPE_DEFAULT;
        if ($this->thread->isSticky) {
            $type = Thread::TYPE_STICKY;
        } elseif ($this->thread->isAnnouncement) {
            $type = Thread::TYPE_ANNOUNCEMENT;
        }

        $availableContentLanguages = LanguageFactory::getInstance()->getContentLanguages();
        if (WCF::getUser()->userID) {
            foreach ($availableContentLanguages as $key => $value) {
                if (!\in_array($key, WCF::getUser()->getLanguageIDs())) {
                    unset($availableContentLanguages[$key]);
                }
            }
        }

        $board = BoardCache::getInstance()->getBoard($this->thread->boardID);
        $boardNodeList = null;
        $boardIDs = [];
        if ($board->getModeratorPermission('canStartAnnouncement')) {
            $boardNodeList = new RealtimeBoardNodeList();
            $boardNodeList->readNodeTree();

            $boardIDs = $this->thread->getAnnouncementBoardIDs();
            if (empty($boardIDs)) {
                $boardIDs[] = $this->thread->boardID;
            }
        }

        WCF::getTPL()->assign([
            'availableContentLanguages' => $availableContentLanguages,
            'board' => $board,
            'boardIDs' => $boardIDs,
            'boardNodeList' => $boardNodeList ? $boardNodeList->getNodeList() : null,
            'thread' => $this->thread,
            'type' => $type,
        ]);

        return WCF::getTPL()->fetch('threadEditorDefault', 'wbb');
    }

    /**
     * @inheritDoc
     */
    public function validate(array &$data)
    {
        // validate topic
        if (isset($data['topic'])) {
            if (!$this->thread->canEdit()) {
                throw new PermissionDeniedException();
            }

            $data['topic'] = StringUtil::trim($data['topic']);
            if (empty($data['topic'])) {
                throw new UserInputException(
                    'topic',
                    WCF::getLanguage()->getDynamicVariable('wcf.global.form.error.empty')
                );
            } elseif (\mb_strlen($data['topic']) > 255) {
                $data['topic'] = \mb_substr($data['topic'], 0, 255);
            }

            $censoredWords = Censorship::getInstance()->test($data['topic']);
            if ($censoredWords) {
                throw new UserInputException(
                    'topic',
                    WCF::getLanguage()->getDynamicVariable(
                        'wcf.message.error.censoredWordsFound',
                        ['censoredWords' => $censoredWords]
                    )
                );
            }
        }

        // validate language
        if (isset($data['languageID'])) {
            $data['languageID'] = isset($data['languageID']) ? \intval($data['languageID']) : 0;

            $availableContentLanguages = LanguageFactory::getInstance()->getContentLanguages();
            if (WCF::getUser()->userID) {
                foreach ($availableContentLanguages as $key => $value) {
                    if (!\in_array($key, WCF::getUser()->getLanguageIDs())) {
                        unset($availableContentLanguages[$key]);
                    }
                }
            }

            if (!\in_array($data['languageID'], \array_keys($availableContentLanguages))) {
                $data['languageID'] = 0;
            }
        }

        // validate thread type
        if (isset($data['type'])) {
            $data['type'] = \intval($data['type']);

            switch ($data['type']) {
                case Thread::TYPE_STICKY:
                    if ($this->thread->isSticky) {
                        break;
                    }

                    if (!$this->thread->getBoard()->getModeratorPermission('canPinThread')) {
                        throw new PermissionDeniedException();
                    }
                    break;

                case Thread::TYPE_ANNOUNCEMENT:
                    // validate board ids
                    if (empty($data['boardIDs']) || !\is_array($data['boardIDs'])) {
                        throw new UserInputException(
                            'boardIDs',
                            WCF::getLanguage()->getDynamicVariable('wcf.global.form.error.empty')
                        );
                    }

                    if (
                        !$this->thread->isAnnouncement
                        && !$this->thread->getBoard()->getModeratorPermission('canStartAnnouncement')
                    ) {
                        throw new PermissionDeniedException();
                    }
                    break;

                default:
                    if ($data['type'] != Thread::TYPE_DEFAULT) {
                        throw new UserInputException(
                            'type',
                            WCF::getLanguage()->getDynamicVariable('wcf.global.form.error.empty')
                        );
                    }
                    break;
            }

            $data['isAnnouncement'] = $data['type'] == Thread::TYPE_ANNOUNCEMENT ? 1 : 0;
            $data['isSticky'] = $data['type'] == Thread::TYPE_STICKY ? 1 : 0;

            unset($data['type']);
        }
    }

    /**
     * @inheritDoc
     */
    public function saveEdit(array $data)
    {
        if (isset($data['languageID']) && $this->thread->languageID != $data['languageID']) {
            $__data['data']['languageID'] = $data['languageID'] ?: null;
        } else {
            unset($data['languageID']);
        }

        $__data = ['data' => $data];

        // clear announcement assignments
        if ($this->thread->isAnnouncement && (!isset($data['isAnnouncement']) || !$data['isAnnouncement'])) {
            $__data['announcementBoardIDs'] = [];
        } elseif (isset($data['isAnnouncement']) && $data['isAnnouncement']) {
            $__data['announcementBoardIDs'] = $data['boardIDs'];
        }

        unset($__data['data']['boardIDs']);

        $action = new ThreadAction([$this->thread], 'update', $__data);
        $action->executeAction();

        // load update thread for comparison
        $thread = new Thread($this->thread->threadID);
        $forceReload = false;
        if ($thread->isAnnouncement != $this->thread->isAnnouncement) {
            $forceReload = true;
        } elseif ($thread->isSticky != $this->thread->isSticky) {
            $forceReload = true;
        }

        return [
            'firstPostID' => $thread->firstPostID,
            'forceReload' => $forceReload,
            'topic' => $thread->topic,
            'languageID' => $thread->languageID,
        ];
    }
}
