/**
 * Provides the custom board icon designer.
 *
 * @author  Alexander Ebert
 * @copyright  2001-2019 WoltLab GmbH
 * @license  WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module  WoltLabSuite/Forum/Acp/Ui/Board/Icon
 */
define(["require", "exports", "tslib", "WoltLabSuite/Core/Core", "WoltLabSuite/Core/Language", "WoltLabSuite/Core/Ui/Color/Picker", "WoltLabSuite/Core/Ui/Dialog", "WoltLabSuite/Core/Ui/Style/FontAwesome", "WoltLabSuite/Core/Dom/Util"], function (require, exports, tslib_1, Core, Language, Picker_1, Dialog_1, UiStyleFontAwesome, Util_1) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.init = void 0;
    Core = tslib_1.__importStar(Core);
    Language = tslib_1.__importStar(Language);
    Picker_1 = tslib_1.__importDefault(Picker_1);
    Dialog_1 = tslib_1.__importDefault(Dialog_1);
    UiStyleFontAwesome = tslib_1.__importStar(UiStyleFontAwesome);
    Util_1 = tslib_1.__importDefault(Util_1);
    class AcpUiBoardIcon {
        /**
         * Initializes the board icon designer.
         */
        constructor() {
            this.currentType = undefined;
            this.icons = new Map();
            const iconContainer = document.getElementById("iconContainer");
            iconContainer.querySelectorAll("dl[data-type]").forEach((container) => {
                const type = container.dataset.type;
                const button = container.querySelector(".button");
                button.addEventListener("click", (event) => this.click(event, type));
                const icon = container.querySelector(".icon");
                this.icons.set(type, {
                    container,
                    icon,
                });
            });
        }
        /**
         * Opens the icon designer for selected type.
         */
        click(event, type) {
            event.preventDefault();
            this.currentType = type;
            Dialog_1.default.open(this);
            const container = this.icons.get(type).container;
            const title = container.querySelector("dt");
            Dialog_1.default.setTitle(this, title.textContent.trim());
        }
        /**
         * Sets the icon name.
         */
        setIcon(iconName) {
            this.icon.textContent = iconName;
            this.renderIcon();
        }
        /**
         * Sets the color, can be either a string or an object holding the
         * individual r, g, b and a values.
         */
        setColor(color) {
            if (typeof color !== "string") {
                color = `rgba(${color.r}, ${color.g}, ${color.b}, ${color.a})`;
            }
            this.color.dataset.color = color;
            this.color.style.setProperty("background-color", color, "");
            this.toggleColor(true);
            this.renderIcon();
        }
        /**
         * Toggles the custom color on or off.
         */
        toggleColor(toggleOn) {
            if (toggleOn !== undefined) {
                this.useColor.checked = toggleOn;
            }
            this.renderIcon();
        }
        /**
         * Renders the custom icon preview.
         */
        renderIcon() {
            const color = this.color.dataset.color;
            const boardIcon = this.content.querySelector(".jsBoardIcon");
            // set icon
            boardIcon.classList.forEach((className) => {
                if (className.indexOf("fa-") === 0) {
                    boardIcon.classList.remove(className);
                }
            });
            boardIcon.classList.add(`fa-${this.icon.textContent}`);
            if (this.useColor.checked) {
                boardIcon.style.setProperty("color", color, "");
                Util_1.default.show(this.colorContainer);
            }
            else {
                boardIcon.style.removeProperty("color");
                Util_1.default.hide(this.colorContainer);
            }
            const colorBoxValue = this.content.querySelector(".colorBoxValue");
            colorBoxValue.style.setProperty("background-color", color, "");
            colorBoxValue.dataset.color = color;
        }
        value(property, newValue) {
            const input = document.querySelector(`input[name="iconData[${this.currentType}][${property}]"]`);
            if (newValue === undefined) {
                return input.value;
            }
            input.value = newValue.toString();
        }
        /**
         * Saves the custom icon design.
         */
        save(event) {
            event.preventDefault();
            const color = this.color.dataset.color;
            const icon = this.icon.textContent;
            const useColor = this.useColor.checked ? 1 : 0;
            this.value("color", color);
            this.value("icon", icon);
            this.value("useColor", useColor);
            const container = this.icons.get(this.currentType).container;
            const boardIcon = container.querySelector(".jsBoardIcon");
            boardIcon.classList.forEach((className) => {
                if (className.indexOf("fa-") === 0) {
                    boardIcon.classList.remove(className);
                }
            });
            boardIcon.classList.add("fa-" + icon);
            if (useColor === 1) {
                boardIcon.style.setProperty("color", color, "");
            }
            else {
                boardIcon.style.removeProperty("color");
            }
            Dialog_1.default.close(this);
        }
        _dialogSetup() {
            return {
                id: "boardIconEditor",
                options: {
                    onSetup: (content) => {
                        this.content = content;
                        this.color = content.querySelector(".colorBoxValue");
                        this.icon = content.querySelector(".jsBoardIconName");
                        this.useColor = content.querySelector(".jsBoardIconUseColor");
                        this.colorContainer = content.querySelector(".jsBoardIconColorContainer");
                        this.useColor.addEventListener("change", () => this.toggleColor());
                        const buttonIconName = content.querySelector(".jsBoardIconName + .button");
                        buttonIconName.addEventListener("click", (event) => {
                            event.preventDefault();
                            UiStyleFontAwesome.open((iconName) => this.setIcon(iconName));
                        });
                        const colorPicker = content.querySelector(".jsColorPicker");
                        const buttonColorPicker = content.querySelector(".jsButtonOpenColorPicker");
                        buttonColorPicker.addEventListener("click", (event) => {
                            event.preventDefault();
                            colorPicker.click();
                        });
                        new Picker_1.default(colorPicker, {
                            callbackSubmit: (color) => this.setColor(color),
                        });
                        const buttonSubmit = content.querySelector(".formSubmit > .buttonPrimary");
                        buttonSubmit.addEventListener("click", (event) => this.save(event));
                    },
                    onShow: () => {
                        this.setIcon(this.value("icon"));
                        this.setColor(this.value("color"));
                        this.toggleColor(Core.stringToBool(this.value("useColor")));
                    },
                    title: Language.get("wbb.acp.board.icon"),
                },
            };
        }
    }
    let acpUiBoardIcon;
    /**
     * Initializes the board icon designer.
     */
    function init() {
        if (!acpUiBoardIcon) {
            acpUiBoardIcon = new AcpUiBoardIcon();
        }
    }
    exports.init = init;
});
