<?php

namespace wbb\acp\form;

use wbb\data\rss\feed\RssFeed;
use wbb\data\rss\feed\RssFeedAction;
use wbb\system\label\object\RssFeedLabelObjectHandler;
use wcf\data\user\User;
use wcf\form\AbstractForm;
use wcf\system\exception\IllegalLinkException;
use wcf\system\WCF;
use wcf\util\ArrayUtil;

/**
 * Shows the RSS feed edit form.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Acp\Form
 */
class RssFeedEditForm extends RssFeedAddForm
{
    /**
     * @inheritDoc
     */
    public $activeMenuItem = 'wbb.acp.menu.link.rssFeed.list';

    /**
     * feed id
     * @var int
     */
    public $feedID = 0;

    /**
     * feed object
     * @var RssFeed
     */
    public $feed;

    /**
     * @inheritDoc
     */
    public function readParameters()
    {
        parent::readParameters();

        if (isset($_REQUEST['id'])) {
            $this->feedID = \intval($_REQUEST['id']);
        }
        $this->feed = new RssFeed($this->feedID);
        if (!$this->feed->feedID) {
            throw new IllegalLinkException();
        }
    }

    /**
     * @inheritDoc
     */
    public function save()
    {
        AbstractForm::save();

        // create board
        $this->objectAction = new RssFeedAction([$this->feed], 'update', [
            'data' => \array_merge($this->additionalFields, [
                'title' => $this->title,
                'url' => $this->url,
                'cycleTime' => $this->cycleTime,
                'maxResults' => $this->maxResults,
                'searchKeywords' => $this->searchKeywords,
                'boardID' => $this->boardID,
                'userID' => $this->user->userID,
                'languageID' => $this->languageID ?: null,
                'closeThread' => $this->closeThread,
                'disableThread' => $this->disableThread,
                'threadTags' => \implode(',', $this->threadTags),
                'useCategoriesAsTags' => $this->useCategoriesAsTags,
            ]),
            'labelIDs' => $this->labelIDs,
        ]);
        $this->objectAction->executeAction();

        // show success message
        WCF::getTPL()->assign('success', true);
    }

    /**
     * @inheritDoc
     */
    public function readData()
    {
        parent::readData();

        if (!\count($_POST)) {
            $this->title = $this->feed->title;
            $this->url = $this->feed->url;
            $this->cycleTime = $this->feed->cycleTime;
            $this->maxResults = $this->feed->maxResults;
            $this->searchKeywords = $this->feed->searchKeywords;
            $this->boardID = $this->feed->boardID;
            $this->languageID = $this->feed->languageID;
            $this->threadTags = ArrayUtil::trim(\explode(',', $this->feed->threadTags));
            $this->useCategoriesAsTags = $this->feed->useCategoriesAsTags;
            $this->closeThread = $this->feed->closeThread;
            $this->disableThread = $this->feed->disableThread;

            $user = new User($this->feed->userID);
            $this->username = $user->username;

            $assignedLabels = RssFeedLabelObjectHandler::getInstance()->getAssignedLabels([$this->feed->feedID], true);
            if (isset($assignedLabels[$this->feed->feedID])) {
                foreach ($assignedLabels[$this->feed->feedID] as $label) {
                    $this->labelIDs[$label->groupID] = $label->labelID;
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function assignVariables()
    {
        parent::assignVariables();

        WCF::getTPL()->assign([
            'action' => 'edit',
            'feedID' => $this->feedID,
            'feed' => $this->feed,
        ]);
    }
}
