<?php

namespace wbb\data\thread;

use wbb\data\board\Board;
use wcf\data\DatabaseObjectList;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\WCF;

/**
 * Represents a list of threads.
 *
 * @author  Marcel Werk
 * @copyright   2001-2019 WoltLab GmbH
 * @license WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package WoltLabSuite\Forum\Data\Thread
 *
 * @method  Thread      current()
 * @method  Thread[]    getObjects()
 * @method  Thread|null getSingleObject()
 * @method  Thread|null search($objectID)
 * @property    Thread[]    $objects
 */
class ThreadList extends DatabaseObjectList
{
    /**
     * @inheritDoc
     */
    public $className = Thread::class;

    /**
     * Excludes read threads from the list.
     *
     * @param   string      $propertyName       name of the thread property relevant for determining read-status
     */
    public function excludeReadThreads($propertyName = 'lastPostTime')
    {
        if (!empty($this->sqlSelects)) {
            $this->sqlSelects .= ',';
        }
        $this->sqlSelects .= "thread.{$propertyName} AS threadNewTime";

        $this->getConditionBuilder()->add(
            "thread.{$propertyName} > ?",
            [VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread')]
        );

        // Prevent collision of multiple conditions using the same alias.
        $aliasSuffix = '_' . $propertyName;

        if (WCF::getUser()->userID) {
            $this->sqlConditionJoins .= "
                LEFT JOIN   wcf" . WCF_N . "_tracked_visit tracked_thread_visit{$aliasSuffix}
                ON          tracked_thread_visit{$aliasSuffix}.objectTypeID = " . VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.thread') . "
                        AND tracked_thread_visit{$aliasSuffix}.objectID = thread.threadID
                        AND tracked_thread_visit{$aliasSuffix}.userID = " . WCF::getUser()->userID . "
                LEFT JOIN   wcf" . WCF_N . "_tracked_visit tracked_board_visit{$aliasSuffix}
                ON          tracked_board_visit{$aliasSuffix}.objectTypeID = " . VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.board') . "
                        AND tracked_board_visit{$aliasSuffix}.objectID = thread.boardID
                        AND tracked_board_visit{$aliasSuffix}.userID = " . WCF::getUser()->userID;

            $this->getConditionBuilder()->add(
                "(thread.{$propertyName} > tracked_thread_visit{$aliasSuffix}.visitTime OR tracked_thread_visit{$aliasSuffix}.visitTime IS NULL)"
            );
            $this->getConditionBuilder()->add(
                "(thread.{$propertyName} > tracked_board_visit{$aliasSuffix}.visitTime OR tracked_board_visit{$aliasSuffix}.visitTime IS NULL)"
            );
        }
    }

    /**
     * Applies the board filter to the internal condition builder
     * instance, optionally removing non-searchable forums.
     *
     * @param       bool         $checkSearchable
     * @param       string[]        $permissions
     * @param       callable        $callback
     * @return      bool         false if there are no accessible boards
     */
    public function applyBoardFilter(
        $checkSearchable = false,
        array $permissions = ['canViewBoard', 'canEnterBoard'],
        ?callable $callback = null
    ) {
        // get board ids
        $boardIDs = Board::filterBoardIDs(Board::getAccessibleBoardIDs($permissions), $checkSearchable, $callback);
        $privateBoardIDs = Board::filterBoardIDs(Board::getPrivateBoardIDs($permissions), $checkSearchable, $callback);

        if (empty($boardIDs) && empty($privateBoardIDs)) {
            $this->getConditionBuilder()->add("1=0");

            return false;
        } else {
            if (empty($privateBoardIDs)) {
                $this->getConditionBuilder()->add("thread.boardID IN (?)", [$boardIDs]);
            } elseif (empty($boardIDs)) {
                $this->getConditionBuilder()->add(
                    "(thread.boardID IN (?) AND thread.userID = ?)",
                    [$privateBoardIDs, WCF::getUser()->userID]
                );
            } else {
                $this->getConditionBuilder()->add(
                    "(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))",
                    [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]
                );
            }
        }

        return true;
    }
}
